package alphabet;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Objects;
import java.util.Set;

public class Wort implements Iterable<Zeichen>, Cloneable, Serializable{

  private static final long serialVersionUID = 1L;
  private List<Zeichen> wort = new ArrayList<>();
	
	public Wort(Zeichen... str) {
		for(Zeichen z: str) {
			this.anfuegen(z);
		}
	}
	
	public Wort(String text) {
	  this.wort = stringAlsWort(text).getWort();
	}
	
  /**
   * Verwandelt uebergebenen String in ein Wort. 
   * Sollte die Zuordnung nicht moeglich sein, wird eine
   * IllegalStateException ausgegeben.
   * @param s umzuwandelnder String
   * @return resultierendes Wort
   * @throws IllegalStateException
   */
  public static Wort stringAlsWort(String s) {
    Wort erg = new Wort();
    if("".equals(s)) {
      return erg;
    }
    String gefunden = "";
    for (int i = 0; i < s.length(); i++) {
      gefunden += s.charAt(i);
      if(Zeichen.istZeichen(gefunden)) {
        Zeichen t = Zeichen.zeichen(gefunden);
        gefunden ="";
        erg.anfuegen(t);
      }
    }
    if (!"".equals(gefunden)) {
      throw new IllegalStateException(s + " nicht vollstaendig "
          + "in Zeichen umgewandelt, fehlt " + gefunden);
    }   
    return erg;
  }
	
	/** Das uebergebene Zeichen wird an das Wort angehaengt.
	 * 
	 * @param z anzuhaengendes Zeichen
	 * @return Wort mit neuem Zeichen am Ende
	 */
	public Wort anfuegen(Zeichen z) {
		this.wort.add(z);
		return this;
	}
	
	 /** Das uebergebene Zeichen wird vor das Wort gehaengt.
   * 
   * @param z neues erstes Zeichen
   * @return Wort mit neuem Anfangszeichen
   */
  public Wort vorne(Zeichen z) {
    this.wort.add(0, z);
    return this;
  }
	
	/** Es wird geprueft, ob das Wort nur aus Terminalzeichen
	 * besteht
	 * @return besteht Wort nur aus Terminalzeichen?
	 */
	public boolean terminal() {
		for(Zeichen z:this.wort) {
			if (z.getClass() == Nichtterminal.class) {
				return false;
			}
		}
		return true;
	}
	
	
	/** Es kann ueber alle Zeichen des Wortes iteriert werden.
	 * 
	 */
	 @Override
	  public Iterator<Zeichen> iterator() {
	    return this.wort.iterator();
	  }
	
	 /** Es werden die Positionen des uebergebenen Zeichens in dem
	  * Wort berechnet, die Positionen werden von 0 an gezaehlt.
	  * @param n zu suchendes Zeichen
	  * @return alles Positionen von n im Wort
	  */
	public List<Integer> positionenVon(Zeichen n){
		List<Integer> erg = new ArrayList<>();
		for(int i = 0; i < this.wort.size(); i++) {
			if (this.wort.get(i) == n) {
				erg.add(i);
			}
		}
		return erg;
	}
	
  /** Im Wort wird an der Position pos das Zeichen durch das Wort w
   * ersetzt. Sollte es die Position nicht geben, wird eine
   * IllegalStateException geworfen.
   * @param pos Position des Zeichens, das ersetzt werden soll
   * @param w einzusetzendes Wort
   * @return dieses Wort mit Ersetzung
   */
	public Wort ersetzeAnPositionMit(int pos, Wort w) {
	  if (this.wort.size() <= pos || pos < 0) {
	    throw new IllegalStateException("Ersetzung an Position " + pos
	        + " in " + this.wort + " nicht machbar.");
	  }
	  this.wort.remove(pos);
	  if(this.wort.size() == pos) {
	    this.wort.addAll(w.wort);
	  } else {
	    this.wort.addAll(pos, w.wort);
	  }
    return this;
	}
	
	/** Das nummer-te Vorkommen des Nichtterminals n wird durch das Wort w
	 * ersetzt, fuer das erste Vorkommen ist nummer=1. Sollte es das 
	 * Vorkommen nicht geben, wird eine IllegalStateException geworfen.
	 * @param n zu ersetzendes Nichtterminal
	 * @param w einzusetzendes Wort
	 * @param nummer das wievielte Vorkommen von n soll ersetzt werden
	 * @return dieses Wort mit vorgenommener Ersetzung
	 * @throws IllegalStateException
	 */
	 public Wort ersetzenVonMitNummer(Nichtterminal n, Wort w, int nummer) {
	    int nr = nummer;
	    for(int i = 0; i < this.wort.size(); i++) {
	      if (this.wort.get(i) == n) {
	        if (nr == 1) {
	          this.wort.remove(i);
	          if(this.wort.size() == i) {
	            this.wort.addAll(w.wort);
	          } else {
	            this.wort.addAll(i, w.wort);
	          }
	          return this;
	        }
	        nr--;
	      }
	    }
	    throw new IllegalStateException(nummer + "-tes Vorkommen von " + n
	        + " in " + this.wort + " existiert nicht.");
	  }
	
	 /** Laenge des Wortes.
	  * 
	  * @return Anzahl der Zeichen des Wortes.
	  */
	public int laenge() {
		return this.wort.size();
	}
	
	/** Ueberprueft, ob das Wort aus genau einem Nichtterminal besteht?
	 * 
	 * @return ist Wort ein Nichtterminal
	 */
	public boolean istNichtterminal() {
	  return this.laenge() == 1 && this.wort.get(0) instanceof Nichtterminal;
	}
	
	/** Falls das Wort nur aus einem Nichtterminal besteht, wird
	 * dieses Nichtterminal zurueckgegeben, sonst wird eine
	 * IllegalStateException geworfen.
	 * @return das eine Nichtterminal aus dem das Wort besteht
   * @throws IllegalStateException	
	 */
	public Nichtterminal alsNichtterminal() {
	  if (this.istNichtterminal()) {
	    return (Nichtterminal) this.wort.get(0);
	  }
	  throw new IllegalStateException(this.wort + " besteht nicht aus"  
        + " genau einem Nichtterminal");
	}
	
	/** Es wird ein neues Wort beginnend ab der Position start inklusive
	 * bis zur Position ende exklusive zurueckgegeben. Sollte eine Position
	 * nicht existieren, wird eine IllegalStateException geworfen.
	 * @param start ab inklusive dieser Position beginnt das neue Wort
	 * @param ende exklusive dieser Position endet das Wort
	 * @return neues Wort (keine tiefe Kopie) ab start bis ende 
	 * @throws IllegalStateException
	 */
	public Wort vonBis(int start, int ende) {
	  Wort erg = new Wort();
	  for(int i = start; i < ende; i++) {
	    erg.anfuegen(this.at(i));
	  }
	  return erg;
	}
	
	/** Gibt das Zeichen an der Position pos zurueck, die Positionen beginnen
	 * bei 0. Existiert die Position nicht, wird eine IllegalStateException 
	 * geworfen.
	 * @param pos Position an der das Zeichen berechnet wird
	 * @return Zeichen an der Position pos
	 * @throws IllegalStateException
	 */
	public Zeichen at(int pos) {
	  if (pos >= this.laenge() || pos < 0) {
	    throw new IllegalStateException(this.wort + " hat kein Zeichen an "  
	        + pos + "-ter Position");
	  }
	  return this.wort.get(pos);
	}
	
	/** Loescht das Zeichen an der Position pos, die Positionen beginnen
   * bei 0. Existiert die Position nicht, wird eine IllegalStateException 
   * geworfen.
   * @param pos Position an der das Zeichen geloescht wird
   * @return Wort ohne geloeschtes Zeichen
   * @throws IllegalStateException
   */
	public Wort loeschen(int pos) {
	  if (pos >= this.laenge() || pos < 0) {
      throw new IllegalStateException(this.wort + " hat kein Zeichen an "  
          + pos + "-ter Position zu loeschen (" + pos + ">=" + this.laenge()+")");
    }
	  this.wort.remove(pos);
	  return this;
	}

	@Override
	public String toString() {
		StringBuilder sb = new StringBuilder();
		for(Zeichen z: this.wort) {
			sb.append(z.toString());
		}
		return sb.toString();
	}
	
	/** String-Repraesentation des Wortes, bei dem Leerzeichen durch
	 *  /space ersetzt werden. 
	 * @return zugehoeriger String mit ersetzten Leerzeichen
	 */
  public String toFile() {
    StringBuilder sb = new StringBuilder();
    for (Zeichen z : this.wort) {
      if (z.toString().equals(" ")) {
        sb.append("/space");
      } else {
        sb.append(z.toString());
      }
    }
    return sb.toString();
  }
	
  /** String-Repraesentation des Wortes, das leere Wort wird durch
   *  ein Epsilon ausgegeben. 
   * @return zugehoeriger String, Leerzeichen als Epsilon
   */
  public String toGui() {
    if (this.laenge() == 0) {
      return "\u03B5";
    }
    return this.toString();
  }
  
	@Override
	public Wort clone() {
	  // return new Wort((Zeichen[])this.wort.toArray()); // geht nicht
	  Wort erg = new Wort();
	    for(Zeichen z: this.wort) {
	      erg.anfuegen(z);
	  }
	    return erg;
	}

	/** Es wird geprueft, ob das Wort ausschliesslich aus Zeichen aus
	 * der uebergebenen Menge m besteht.
	 * @param m Menge von Nichtterminalzeichen
	 * @return besteht Wort nur aus Zeichen aus m?
	 */
  public boolean bestehtAus(Set<Nichtterminal> m) {
    for(Zeichen z:this.wort) {
      if (z.getClass() == Terminal.class) {
        return false;
      }
      if (!m.contains(z)) {
        return false;
      }
    }
    return true;
  }
  
  /** Es wird die erste Position eines Nichtterminals im Wort
   * zurueckgegegeben. Existiert diese Position nicht, ist das
   * Ergebnis -1.
   * @return erster Position eines Nichtterminals
   */
  public int positionErstesNichtterminal() {
    for (int i = 0; i < this.laenge(); i++) {
      if (this.wort.get(i) instanceof Nichtterminal) {
        return i;
      }
    }
    return - 1;
  }
  
  /** Prueft ob Wort ausschließlich aus Nichtterminalen besteht.
   * 
   * @return nur aus Nichtterminalen?
   */
  public boolean ausNichtTerminalen() {
    for (Zeichen z: this.wort) {
      if (! (z instanceof Nichtterminal)) {
        return false;
      }
    }
    return true;
  }

  @Override
  public int hashCode() {
    return Objects.hash(wort);
  }

  @Override
  public boolean equals(Object obj) {
    if (this == obj)
      return true;
    if (obj == null)
      return false;
    if (getClass() != obj.getClass())
      return false;
    Wort other = (Wort) obj;
    return Objects.equals(wort, other.wort);
  }

  public List<Zeichen> getWort() {
    return wort;
  }

  public void setWort(List<Zeichen> wort) {
    this.wort = wort;
  }
}
