package endlicherAutomat;

import java.io.BufferedWriter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import alphabet.Terminal;
import util.DeepClone;
import util.Paar;
import zustand.Zustand;

public class AutomatUeberfuehrungsfunktion 
    implements DeepClone<AutomatUeberfuehrungsfunktion>{
  private static final long serialVersionUID = 1L;
  /** Zum Zustand und Terminal wird die Menge der moeglichen
   * Folgezustaende festgehalten. List, um Reihenfolge beizubehalten.
   */
   private Map<Paar<Zustand, Terminal>, List<Zustand>> funktion = new HashMap<>();
   private Map<Zustand, List<Zustand>> epsilon = new HashMap<>();
   
   public void add(Zustand alt, Terminal lesen, Zustand neu) {
     Paar<Zustand, Terminal> links = new Paar<>(alt,lesen);
     List<Zustand> bisher = this.funktion.get(links);
     if (bisher == null) {
       List<Zustand> tmp = new ArrayList<>();
       tmp.add(neu);
       this.funktion.put(links, tmp);
     } else {
       if(! bisher.contains(neu)) {
         bisher.add(neu);
       }
     }
   }
   
   public void addEpsilon(Zustand alt, Zustand neu) {
     List<Zustand> bisher = this.epsilon.get(alt);
     if (bisher == null) {
       List<Zustand> tmp = new ArrayList<>();
       tmp.add(neu);
       this.epsilon.put(alt, tmp);
     } else {
       if(! bisher.contains(neu)) {
         bisher.add(neu);
       }
     }
   }
   
   public boolean hatEpsilonverbindung(Zustand z1, Zustand z2) {
     if (this.epsilon.get(z1) == null) {
       return false;
     }
     return this.epsilon.get(z1).contains(z2);
   }
   
   public void removeEpsilon(Zustand alt, Zustand weg) {
     List<Zustand> bisher = this.epsilon.get(alt);
     if (bisher != null) {
       if(bisher.size() == 1) {
         this.epsilon.remove(alt);
       } else {
         bisher.remove(weg);
       }
     }
   }
   
   public void remove(Zustand alt, Terminal lesen, Zustand weg) {
     Paar<Zustand, Terminal> links = new Paar<>(alt,lesen);
     List<Zustand> bisher = this.funktion.get(links);
     if (bisher != null) {
       if(bisher.size() == 1) {
         this.funktion.remove(links);
       } else {
         bisher.remove(weg);
       }
     }
   }
   /** Prueft, ob es mindestens einen Uebergang fuer den Zustand und das
    * Terminal gibt, Epsilon-Uebergaenge werden nicht beruecksichtigt
    * @param zu zu pruefendes Terminal
    * @param ze dazu zu pruefender Zustand
    * @return existiert mindestens ein Uebergang fuer (zu,ze)?
    */
   public boolean istDefiniert(Zustand zu, Terminal ze) {
     return this.funktion.get(new Paar<>(zu,ze)) != null;
   }
   
   /** Prueft, ob es genau einen Uebergang fuer den Zustand und das
    * Terminal gibt, Epsilon-Uebergaenge werden nicht beruecksichtigt
    * @param zu zu pruefendes Terminal
    * @param ze dazu zu pruefender Zustand
    * @return existiert genau ein Uebergang fuer (zu,ze)?
    */
   public boolean istEindeutig(Zustand zu, Terminal ze) {
     return this.istDefiniert(zu, ze)
         && this.funktion.get(new Paar<>(zu,ze)).size() == 1;
   }
   
   /** Berechnet fuer einen uebergebenen Zustand und ein Terminal die
    * Menge der so erreichbaren Folgezustaende, dabei werden Epsilon-
    * Schritte vor und nach der Nutzung des Terminals beruecksichtigt.
    * Sollte es keinen Uebergang geben, ist der leere Liste das Ergebnis.
    * @param alt Ausgangszustand
    * @param lesen zu verarbeitendes Terminal
    * @return Menge der mit dem Terminal erreichbaren Folgezustaende
    */
   public List<Zustand> moeglicheFolgezustaende(Zustand alt, Terminal lesen) {
     List<Zustand> ergebnis =  new ArrayList<>();
     List<Zustand> start = this.mitEpsilonErreichbar(alt);
     //System.out.println("start: " + start);
     //boolean gefunden = false;
     for(Zustand zalt: start) {
       List<Zustand> erg = this.funktion.get(new Paar<>(zalt, lesen));
       if (erg != null && !erg.isEmpty()) {
         List<Zustand> erg2 = new ArrayList<>();
         for(Zustand zst: erg) {
           List<Zustand> eps = this.mitEpsilonErreichbar(zst);
           for(Zustand zst2: eps) {
             if (!erg2.contains(zst2)) {
               erg2.add(zst2);
             }
           }
         }
         for(Zustand zst: erg2) {
           if (!ergebnis.contains(zst)) {
             ergebnis.add(zst);
             //gefunden = true;
           }
         }
       }
     }
//     if(! gefunden) {
//       System.err.println("WARNING: gibt keinen Nachfolgezustand zu " 
//           + alt + "," + lesen + ":\n"); // + this);
//     }
     return ergebnis;
   }
   
   /** Berechnet die Mene der Zustaende, die aus dem uebergebenen 
    * Zustand und dem Zeichen erreicht werden kann
    * @param z Ausgangszustand
    * @param t zu verarbeitendes Zeichen
    * @return Menge der moeglichen Folgezustaende
    */
   public List<Zustand> direktErreichbar(Zustand z, Terminal t){
     List<Zustand> erg = this.funktion.get(new Paar<>(z,t));
     if (erg == null) {
       return new ArrayList<>();
     }
     return erg;
   }
   
   /** Berechnet die Menge aller Zustaende, die mit Epsilon-
    * Uebergaengen aus dem uebergebenen zustand erreichbar
    * sind
    * @param z Ausgangszustand
    * @return aus z mit Episilon-Uebergaengen erreichbar
    */
   public List<Zustand> mitEpsilonErreichbar(Zustand z){
     List<Zustand> erg = new ArrayList<>();
     erg.add(z);
     int size = 0;
     while (erg.size() != size) {
       List<Zustand> tmp = new ArrayList<>();
       tmp.add(z);
       size = erg.size();
       for(Zustand zu: erg) {
         if (epsilon.get(zu) != null) {
           for(Zustand zst: epsilon.get(zu)) {
             if (!tmp.contains(zst)) {
               tmp.add(zst);
             }
           }
         }
       }
       for(Zustand zu: tmp) {
         if (!erg.contains(zu)) {
           erg.add(zu);
         }
       }
       erg = tmp;
     }
     return erg;
   }
   
   /** Ersetzt alle Epsilon-Uebergaenge durch Kanten mit Terminal,
    * die durch vorherige oder nachfolgende Epsilon-Uebergaenge
    * erreichbar sind.
    * @return sprachaequivalent umgeformte Ueberfuehrungsfunktion ohne
    *         Epsilon-Uebergaenge
    */
   public AutomatUeberfuehrungsfunktion epsilonEntfernen() {
     //List<Zustand> abgearbeitet = new ArrayList<>();
     Map<Paar<Zustand, Terminal>, List<Zustand>> hinzu = new HashMap<>();
     for(Zustand zu: this.epsilon.keySet()) { // zu jedem Zustand mit Epsilon-Uebergaengen
       List<Zustand> erreichbar = this.mitEpsilonErreichbar(zu); // berechne alle mit Epsilon erreichbaren Zustaende
       for(Zustand err: erreichbar) {
         if(!err.equals(zu)) {
           for (Paar<Zustand,Terminal> pa: this.funktion.keySet()) { // fuer diese erreichbaren Zustaende
             if(pa.getLinks().equals(err)) {  // schaue existierende Transitionen an
               List<Zustand> neu = this.moeglicheFolgezustaende(err, pa.getRechts()); // berechne dazu alle mit dem Terminal erreichbaren Zustaende
               hinzu.put(new Paar<>(zu, pa.getRechts()), neu); // und ergaenze diese fuer den Ausgangszustand
             }
           }
         }
       }
     }
         
     for (Paar<Zustand,Terminal> pa: hinzu.keySet()) {
       for(Zustand next: hinzu.get(pa)) {
         this.add(pa.getLinks(), pa.getRechts(), next);
       }
     }
     
     // Berechne fuer existierende Transitionen die zusaetzlich mit 
     // Epsilon-Schritten erreichbaren Zustaende
     for (Paar<Zustand,Terminal> pa: this.funktion.keySet()) {
       List<Zustand> alle = this.moeglicheFolgezustaende(pa.getLinks()
           , pa.getRechts());
       if(alle == null) {
         System.err.println(pa + " hat null Folgezustaende ");
       }
       if(alle.size() == 0) {
         System.err.println(pa + " hat 0 Folgezustaende ");
       }
       this.funktion.put(pa, alle);
     }
    
     this.epsilon = new HashMap<>();
     return this;
   }
   
   @Override
   public AutomatUeberfuehrungsfunktion clone() {
     return this.deepClone();
   }
   
  @Override
  public int hashCode() {
    return Objects.hash(funktion);
  }

  @Override
  public boolean equals(Object obj) {
    if (this == obj)
      return true;
    if (obj == null)
      return false;
    if (getClass() != obj.getClass())
      return false;
    AutomatUeberfuehrungsfunktion other = (AutomatUeberfuehrungsfunktion) obj;
    return Objects.equals(funktion, other.funktion);
  }

  @Override
   public String toString() {
     StringBuilder sb = new StringBuilder("Ueberfuehrungsfunktion:\n");
     for(Paar<Zustand, Terminal> alt: this.funktion.keySet()) {
       List<Zustand> neu = this.funktion.get(alt);
       if(neu == null) {
         System.err.println("null bei "+ alt);
       }
       for(Zustand n: neu) {
         sb.append(" ueber(" + alt.links +", " + alt.rechts + ") = "
             + n +"\n");
       }
     }
     for(Zustand alt: this.epsilon.keySet()) {
       List<Zustand> neu = this.epsilon.get(alt);
       if(neu == null) {
         System.err.println("null bei "+ alt);
       }
       for(Zustand n: neu) {
         sb.append(" ueber(" + alt +", \u03B5) = "
             + n +"\n");
       }
     }
     return sb.toString();
   }

   public void inDatei(BufferedWriter writer) throws IOException {
     @SuppressWarnings("unchecked")
    List<String>[] spalten = new List[3];
     int[] max = new int[3];
     for(int i = 0; i < 3; i++) {
       spalten[i] = new ArrayList<>();
       max[i] = 0;
     }
     
     for(Paar<Zustand, Terminal> alt: this.funktion.keySet()) {
       List<Zustand> neu = this.funktion.get(alt);
       for(Zustand n: neu) {
         spalten[0].add(alt.getLinks().toString());
         spalten[1].add(alt.getRechts().toString());
         spalten[2].add(n.toString());
       }
     }
     
     for(int i = 0; i < 3; i++) {
       for(String s: spalten[i]) {
         if(s.length() > max[i]) {
           max[i] = s.length();
         }
       }
     }
     
     for(int z = 0; z < spalten[0].size(); z++) {
       StringBuilder sb = new StringBuilder();
       for(int i = 0; i < 3; i++) {
         sb.append(spalten[i].get(z));
         sb.append(leerzeichen(max[i] - spalten[i].get(z).length() + 1));
       }
       sb.append("\n");
       writer.write(sb.toString());
     }  
     
     if(this.epsilon.isEmpty()) {
       return;
     }
     
     max = new int[3];
     for(int i = 0; i < 3; i++) {
       spalten[i] = new ArrayList<>();
       max[i] = 0;
     }
     
     for(Zustand alt: this.epsilon.keySet()) {
       List<Zustand> neu = this.epsilon.get(alt);
       for(Zustand n: neu) {
         spalten[0].add(alt.toString());
         spalten[1].add("/eps");
         spalten[2].add(n.toString());
       }
     }
     
     for(int i = 0; i < 3; i++) {
       for(String s: spalten[i]) {
         if(s.length() > max[i]) {
           max[i] = s.length();
         }
       }
     }
     
     for(int z = 0; z < spalten[0].size(); z++) {
       StringBuilder sb = new StringBuilder();
       for(int i = 0; i < 3; i++) {
         sb.append(spalten[i].get(z));
         sb.append(leerzeichen(max[i] - spalten[i].get(z).length() + 1));
       }
       sb.append("\n");
       writer.write(sb.toString());
     }   
   }
   
   public String leerzeichen(int anzahl) {
     StringBuilder sb = new StringBuilder();
     for( int i = 0; i < anzahl; i++) {
       sb.append(" ");
     }
     return sb.toString();
   }

   public void inDateiFormatiert(BufferedWriter writer, int altsp
       , int lesen, int neusp, String trenn1
       , String trenn2, String trenn3)
           throws IOException {
     @SuppressWarnings("unchecked")
    List<String>[] spalten = new List[3];
     int[] max = new int[3];
     for(int i = 0; i < 3; i++) {
       spalten[i] = new ArrayList<>();
       max[i] = 0;
     }
     
     for(Paar<Zustand, Terminal> alt: this.funktion.keySet()) {
       List<Zustand> neu = this.funktion.get(alt);
       for(Zustand n: neu) {
         spalten[0].add(alt.getLinks().toString());
         spalten[1].add(alt.getRechts().toString());
         spalten[2].add(n.toString());
       }
     }
     
     for(int i = 0; i < 3; i++) {
       for(String s: spalten[i]) {
         if(s.length() > max[i]) {
           max[i] = s.length();
         }
       }
     }
     
     String[] trenner = {trenn1, trenn2, trenn3};
     
     for(int z = 0; z < spalten[0].size(); z++) {
       StringBuilder sb = new StringBuilder();
       for(int zz = 0;  zz < 3; zz++) {
         sb.append(spalten[zz].get(z) + trenner[zz]);
       }
       sb.append("\n");
       writer.write(sb.toString());
     } 
   }
   
   public boolean istOhneEpsilon() {
     return this.epsilon.size() == 0;
   }
   
   public boolean istDeterministisch() {
     for(Paar<Zustand, Terminal> alt: this.funktion.keySet()) {
       List<Zustand> neu = this.funktion.get(alt);
       if (neu == null || neu.size() != 1) {
         return false;
       }
     }
     return true;
   }

  public Map<Paar<Zustand, Terminal>, List<Zustand>> getFunktion() {
    return funktion;
  }

  public void setFunktion(
      Map<Paar<Zustand, Terminal>, List<Zustand>> funktion) {
    this.funktion = funktion;
  }

  public Map<Zustand, List<Zustand>> getEpsilon() {
    return epsilon;
  }

  public void setEpsilon(Map<Zustand, List<Zustand>> epsilon) {
    this.epsilon = epsilon;
  }
}
