package endlicherAutomat;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.IOException;
import java.io.Serializable;
import java.nio.charset.Charset;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;

import alphabet.Nichtterminal;
import alphabet.Terminal;
import alphabet.Wort;
import alphabet.Zeichen;
import grammatik.KontextfreieGrammatik;
import grammatik.KontextfreieRegel;
import regulaererAusdruck.Klammern;
import regulaererAusdruck.Leer;
import regulaererAusdruck.Oder;
import regulaererAusdruck.Punkt;
import regulaererAusdruck.RegulaererAusdruck;
import regulaererAusdruck.Stern;
import util.DeepClone;
import util.Eingabe;
import util.Paar;
import zustand.Zustand;

public class EndlicherAutomat implements DeepClone<EndlicherAutomat>
  , Serializable{

  private static final long serialVersionUID = 5094610689453445770L;
  protected List<Zustand> zustaende = new ArrayList<>();
  protected List<Zustand> endzustaende = new ArrayList<>();
  protected List<Terminal> alphabet = new ArrayList<>();
  protected Zustand start;
  protected AutomatUeberfuehrungsfunktion ueber;
  protected AutomatKonfiguration konfiguration;
  protected boolean[][] aequivalent;
  // Zustaende, die fuer eine Minimierung genutzt wurden
  protected List<Zustand> minimierungzustaende = new ArrayList<>();
  
  public EndlicherAutomat() {
    this.ueber = new AutomatUeberfuehrungsfunktion();
  }
  
  /** Nach der Methode befindet sich der Zustand e in der
   * Endzustansmenge
   * @param e als Endzustand einzutragen
   * @return der evtl. um Endzustand ergaenzte Automat
   */
  public EndlicherAutomat addEndzustand(Zustand e) {
/////    e.setEnde(true);
    if(! this.endzustaende.contains(e)) {
      this.endzustaende.add(e);
    }
    return this;
  }
  
  /** Nach der Methode befindet sich ein Zustand mit dem Namen e in der
   * Endzustansmenge, insofern es bereits ein Zustand mit disem Namen
   * in der Grundmenge der Zustaende gibt
   * @param e Name des Zustands der als Endzustand einzutragen ist
   * @return der evtl. um Endzustand ergaenzte Automat
   */
  public EndlicherAutomat addEndzustand(String e) {
    this.addEndzustand(Zustand.zustand(e));
    return this;
  }
  
  /** Alle in der Datei enthaltenen Automateninformationen werden in
   * dieses Objekt (ueblicherweise frisch erzeugt) eingetragen
   * @param datei Datei mit neuen zu lesenden Automateninformationen
   */
  public void dateiEinlesen(String datei) {
    File file = new File(datei);
      if (file.exists() && file.canRead()) {
        Path path = Paths.get(datei);
        try (BufferedReader reader = Files
            .newBufferedReader(path, Charset.forName("UTF-8"))) {
          String currentLine = null;
          StringBuilder sb = new StringBuilder();
          while ((currentLine = reader.readLine()) != null) {
            sb.append(currentLine + "\n");         
          }
          // System.out.println(sb);
          this.stringAlsAutomat(sb.toString());
        } catch(Exception e) {
          System.err.println("WARNING: " + datei 
              + " existiert: " + file.exists()
              + " lesbar: " + file.canRead()
              + " meldung: " + e
          );
        }
      }else {
        System.err.println("WARNING: " + datei 
            + " existiert: " + file.exists()
            + " lesbar: " + file.canRead()
        );
  }
  }
  
  /** Der aktuelle Automat wird in der Datei gespeichert.
   * 
   * @param datei Name der Datei, in der der Automat gespeichert wird
   */
  public void dateiSpeichern(String datei) {
    File file = new File(datei);
    try {
      if(!file.exists()) {
        file.createNewFile();
      }
    } catch (IOException e1) {
      throw new IllegalStateException("Datei " + datei 
          + " nicht erzeugbar: " + e1);
    }
    Path path = Paths.get(datei);
    try (BufferedWriter writer = Files.newBufferedWriter(path, Charset.forName("UTF-8"))) {
      StringBuilder sb = new StringBuilder("Z: ");
      for(Zustand z: this.zustaende) {
         sb.append(z.toString() + " ");
      }
      sb.append("\nE: ");
      for(Zustand z: this.endzustaende) {
        sb.append(z.toString() + " ");
     }
      writer.write(sb.toString() +"\n");
      
      StringBuilder sb2 = new StringBuilder("A: ");
      for(Terminal z: this.alphabet) {
          sb2.append(z.toString() + " ");
      }
      writer.write(sb2.toString() + "\n");
      writer.write("S: " + this.start +"\n");
      this.ueber.inDatei(writer);
      
    } catch(Exception e) {
      System.err.println("Warnung: Problem beim Schreiben in die Datei " 
          + datei + ": " + e);
      e.printStackTrace();
    }
  }
  
  /** Alle im String enthaltenen Automateninformationen werden in
  * dieses Objekt (ueblicherweise frisch erzeugt) eingetragen
  * @param gram String mit neuen zu lesenden Automateninformationen
  */
  public void stringAlsAutomat(String gram) {
    String[] zeilen = gram.split("\\n");
    this.ueber = new AutomatUeberfuehrungsfunktion();
    this.zustaende = new ArrayList<>();
    this.endzustaende = new ArrayList<>();
    Terminal.reset();
    this.alphabet = new ArrayList<>();

    for(String s:zeilen) {
      this.verarbeiteZeile(s);    
    }
  }
  
  private void verarbeiteZeile(String s) {
    String trim =s.split("//")[0].trim();
    if(trim.length() == 0) {
      return;
    }
    if(trim.startsWith("%")) {
      return;
    }
    if(trim.startsWith("Z:")) {
      this.neueZustaende(trim.substring(2).trim());
      return;
    }
    if(trim.startsWith("E:")) {
      this.neueEndzustaende(trim.substring(2).trim());
      return;
    }
    if(trim.startsWith("A:")) {
      this.neueZeichen(trim.substring(2).trim());
      return;
    }
    
    if(trim.startsWith("S:")) {
      Zustand tmp = this.zustandFallsExistert(trim.substring(2).trim());
      if (!this.zustaende.contains(tmp)) {
        throw new IllegalStateException(trim.substring(2).trim() 
            + " muss als Endzustand in der Zustandsliste stehen");
      }
      this.start = tmp;
      return;
    }
    
    //Ueberfuehrungsfunktion
    String[] alle = trim.split("\\s+");
    if(alle.length != 3) {
      System.err.println("Warnung: " + trim + " ignoriert"
          + ", Ueberfuehrungsfunktion eines Automaten benoetigt 3 Elemente");
    } else {
      if(!Zustand.istZustand(alle[0])) {
        System.err.println("Warnung: " + trim + " ignoriert, "
            + alle[0] + " ist kein Zustand");
        return;
      }
      if(!Zustand.istZustand(alle[2])) {
        System.err.println("Warnung: " + trim + " ignoriert, "
            + alle[2] + " ist kein Zustand");
        return;
      }
      if(alle[1].equals("/eps")) {
        this.addEpsilonueberfuehrung(alle[0], alle[2]);
      } else {
        if(!Terminal.istTerminal(alle[1])) {
          System.err.println("Warnung: " + trim + " ignoriert, "
              + alle[1] + " ist kein erlaubtes Zeichen");
          return;
        }
        this.addUeberfuehrung(alle[0], alle[1], alle[2]);
      }
    }   
  }
  
  private Zustand zustandFallsExistert(String name) {
    if (!Zustand.istZustand(name)) {
      throw new IllegalStateException(name 
          + " muss vor der Nutzung in der Zustandsliste stehen");
    }
    return Zustand.zustand(name);
  }
  
  private void neueZustaende(String s) {
    String[] alle = s.split("\\s+");
    for(String st:alle) {
      if (! Zustand.istZustand(st)) {
        this.zustaende.add(Zustand.zustand(st));
      } else {
        this.zustaende.add(Zustand.zustand(st));
      }
    }
  }
  
  private void neueEndzustaende(String s) {
    String[] alle = s.split("\\s+");
    for(String st:alle) {
      if (! Zustand.istZustand(st)) {
        throw new IllegalStateException("Endzustand muss vorher als Zustand"
            + " definiert sein: " + st);
      } else {
        Zustand z = Zustand.zustand(st);
    /////        z.setEnde(true);
        this.endzustaende.add(z);
      }
    }
  }
  
  private void neueZeichen(String s) {
    String[] alle = s.split("\\s+");
    for(String st:alle) {
      if (! Zeichen.istZeichen(st)) {
        this.alphabet.add(Terminal.terminal(st));
      } else {
        this.alphabet.add(Terminal.terminal(st));
      }
    }
  }
  
  /** Uebergebener String wird in Wort gewandelt und zur
   * Ausgangskonfiguration umgewandelt.
   * @param init abzuarbeitender String
   */
  public void bearbeite(String init) {
    this.bearbeite(new Wort(init));
  }
  
  /** Uebergebenes Wort wird zur
   * Ausgangskonfiguration umgewandelt.
   * @param init abzuarbeitendes Wort
   */
  public void bearbeite(Wort init) {
    this.konfiguration = new AutomatKonfiguration(this.start, init.clone());
  }
  
  /** Ausgehend von der mit bearbeite uebergebenen Startkonfiguration
   * wird der Automat schrittweise abgearbeitet, das Ergebnis zeigt,
   * ob das abzuarbeitendes Wort akzeptiert wird. Dabei kann die
   * Ausgabe und die schrittweise interaktive Ausfuehrung ein- und
   * ausgeschaltet werden
   * @param zeigeDetails sollen alle Konfigurationen ausgegeben werden
   * @param schrittweise soll nach jedem Schritt auf Nutzungsreaktion
   *        (Eingabetaste) gewartet werden
   * @return wurde Wort der Ausgangskonfiguration akzeptiert
   */
  public boolean ausfuehrenSimulation(boolean zeigeDetails, boolean schrittweise) {
    if(this.konfiguration == null) {
      throw new IllegalStateException("Vor Berechnung ein Wort zur "
          + "Bearbeitung uebergeben");
    }
    if (zeigeDetails) {
      System.out.println("Start: " + this.konfiguration);
    }
    while(!this.konfiguration.getRest().equals(new Wort())) {
      this.schritt();
      if (zeigeDetails) {
        System.out.println(this.konfiguration);
      }
      if(schrittweise) {
        System.out.println("Weiter mit <Return>");
        Eingabe.leseString();  
      }
    }
    if (zeigeDetails) {
      System.out.println("Ende: " + this.konfiguration);
    }
    List<Zustand> moeglichesEnde = this.ueber.mitEpsilonErreichbar(this.konfiguration.getZustand());
    if (this.endzustaende.contains(this.konfiguration.getZustand())) {
      return true;
    }
    for(Zustand z: moeglichesEnde){
      if (this.endzustaende.contains(z)) {
        System.out.println("akzeptiert mit " + this.konfiguration.getZustand()
         + " -*-> " + z);
        return true;
      }
    }
    return false;
  }
  
  /**
   * Ausgehend von der mit bearbeite uebergebenen Startkonfiguration
   * wird der Automat schrittweise abgearbeitet, das Ergebnis zeigt,
   * ob das abzuarbeitendes Wort akzeptiert wird, die Konfigurationen
   * werden ausgegeben.
   * @return wurde Wort der Ausgangskonfiguration akzeptiert
   */
  public boolean ausfuehrenSchrittweise() {
    return this.ausfuehrenSimulation(true);
  }
  
  /**
   * Ausgehend von der mit bearbeite uebergebenen Startkonfiguration
   * wird der Automat schrittweise abgearbeitet, das Ergebnis zeigt,
   * ob das abzuarbeitendes Wort akzeptiert wird, dabei kann festgelegt
   * werden, ob alle Konfigurationen ausgegeben werden sollen
   * @param zeigeDetails sollen alle Konfigurationen ausgegeben werden
   * @return wurde Wort der Ausgangskonfiguration akzeptiert
   */
  public boolean ausfuehrenSimulation(boolean zeigeDetails) {
    return this.ausfuehrenSimulation(zeigeDetails, false);
  }
  
  /** Ueberprueft ob, der uebergebene String als Wort vom 
   * Automaten akzeptiert wird
   * @param eingabe zu ueberpruefender String
   * @return wird Wort akzeptiert?
   */
  public boolean akzeptieren(String eingabe) {
    return this.akzeptieren(new Wort(eingabe));
  }
  
  /** Ueberprueft ob das uebergebene Wort vom 
   * Automaten akzeptiert wird
   * @param eingabe zu ueberpruefendes Wort
   * @return wird Wort akzeptiert?
   */
  public boolean akzeptieren(Wort eingabe) {
    return this.akzeptieren(eingabe, true);
  }
  /**Ueberprueft ob das uebergebene Wort vom 
   * Automaten akzeptiert wird, dabei kann festgelegt
   * werden, ob alle Konfigurationen ausgegeben werden sollen
   * @param eingabe zu ueberpruefendes Wort
   * @param zeigeDetails sollen alle Konfigurationen ausgegeben werden
   * @return wird Wort akzeptiert?
   */
  public boolean akzeptieren(Wort eingabe, boolean zeigeDetails) {
    EndlicherAutomat checker = this.clone();
    if(!checker.istOhneEpsilon()) {
      checker.epsilonEntfernen();
    }
    if (!checker.istVollstaendig()) {
      checker.vervollstaendigen();
    }
    if(!checker.istDeterministisch()) {
      checker.deterministisch();
    }
    checker.bearbeite(eingabe);
    return checker.ausfuehrenSimulation(zeigeDetails);
  }
  
  /** Die Epsilon-Uebergaenge dieses Automaten werden sprachaequivalent
   * entfernt
   * @return dieser Automat ohne Epsilon-Uebergaenge
   */
  public EndlicherAutomat epsilonEntfernen() {
    //System.out.println("epsilonEntfernen");
    List<Zustand> weitereEndzustaende = new ArrayList<>();
    for(Zustand zu: this.zustaende) {
      //boolean gefunden = false;
      for(Zustand ende: this.endzustaende) {
        if (this.ueber.mitEpsilonErreichbar(zu).contains(ende)
            && !weitereEndzustaende.contains(zu)) {
          weitereEndzustaende.add(zu);
          //gefunden = true;
        }
      }
    }

    this.ueber.epsilonEntfernen();
    
    for(Zustand zu: weitereEndzustaende) {
      //zu.setEnde(true);
      //System.out.println(zu + " 2:\n" + this + "\n");
      if (!this.endzustaende.contains(zu)) {
        this.endzustaende.add(zu);
      }
    }
    return this;
  }
  
  /** Das uebergebene Wort wird schrittweise interaktiv abgearbeitet.
   * Gibt es Auswahlmoeglichkeiten, wie mehrere Epsilon-Schritte oder
   * verschiedene moegliche Folezustaende, werden die Moeglichkeiten
   * in einem Nutzungsdialog angeboten. Besteht keine Auswahlmoeglichkeit
   * schreitet das Programm direkt oran.
   * @param w abzuarbeitendes Wort
   * @return wurde w vom Automaten in der ausgewaehlten 
   * Abarbeitungsreihenfolge akzeptiert (false bedeutet nicht, 
   * dass das Wort garantiert nicht akzeptiert werden kann.
   */
  public boolean wortAbarbeiten(String w) {
    return this.wortAbarbeiten(new Wort(w));
  }
  
  public boolean wortAbarbeiten(Wort wort) {
    this.bearbeite(wort);
    boolean abbruch = false;
    Zustand zustand = this.start;
    while(wort.laenge() !=0 && !abbruch) {
      System.out.println("Eingabe: " + wort + "  Zustand: " + zustand);
      List<Zustand> mitepsErreichbar = this.ueber
          .mitEpsilonErreichbar(zustand);
      mitepsErreichbar.remove(zustand);
      List<Zustand> einSchritt = this.ueber
          .direktErreichbar(zustand, (Terminal)wort.at(0));
      //System.out.println(mitepsErreichbar + " : " + einSchritt);
      if(mitepsErreichbar.isEmpty() && einSchritt.isEmpty()) {
        System.out.println("kein Folgeschritt moeglich.");
        abbruch = true;
        return false;
      }
      boolean eps = false;
      boolean step = false;
      if(!mitepsErreichbar.isEmpty() && !einSchritt.isEmpty()) {
        System.out.print("Epsilonschritt (e) oder erstes Zeichen abarbeiten (a): ");
        while(!eps && !step) {
          String in = Eingabe.leseString();
          if(in.length() > 0 && in.startsWith("e")) {
            eps = true;
          }
          if(in.length() > 0 && in.startsWith("a")) {
            step = true;
          }
        }       
      }
      if(!mitepsErreichbar.isEmpty() && (einSchritt.isEmpty() || eps)) {
        if(mitepsErreichbar.size() == 1) {
          zustand = mitepsErreichbar.get(0);
        } else {
          zustand = this.zustandsauswahl(mitepsErreichbar);
        }
      }
      if((mitepsErreichbar.isEmpty() || step )&& !einSchritt.isEmpty()) {
        if(einSchritt.size() == 1) {
          zustand = einSchritt.get(0);
        } else {
          zustand = this.zustandsauswahl(einSchritt);
        }
        wort = wort.vonBis(1, wort.laenge());
      }
    }
    if (this.endzustaende.contains(zustand)) {
      System.out.println("Wort wurde auf diesem Weg akzeptiert");
      return true;
    }
    System.out.println("Wort wurde auf diesem Weg nicht akzeptiert");
    return false;   
  }
  
  
  private EndlicherAutomat schritt() {
    if(this.konfiguration.getRest().equals(new Wort())) {
      System.err.println("Nach abgearbeiteten Wort folgt keinen Schritt mehr.");
      return this;
    }
    List<Zustand> neuZu = this.ueber.moeglicheFolgezustaende(this.konfiguration.getZustand()
        , (Terminal)this.konfiguration.getRest().at(0));

    if(neuZu.size() == 0) {
      System.err.println("kein Folgezustand zu: " + this.konfiguration.getZustand()
      + "," + this.konfiguration.getRest().at(0));
      return this;
    }
    Zustand next;
    if (neuZu.size() == 1) {
      next = neuZu.get(0);
    } else {
      next = this.zustandsauswahl(neuZu);
    }
    this.konfiguration.setZustand(next);
    this.konfiguration.setRest(this.konfiguration.getRest().loeschen(0));
    return this;
  }
  
  private Zustand zustandsauswahl(List<Zustand> neuZu) {
    System.out.println("Welchen der moeglichen Folgezustaende?");
    for(int i = 0; i < neuZu.size(); i++) {
      System.out.println("(" + i + ") " + neuZu.get(i));
    }
    int ein = -1;
    while(ein < 0 || ein > neuZu.size() - 1) {
      ein = Eingabe.leseInt();
    }
    return neuZu.get(ein);
  }
  
  /** Dieser Automat wird vervollstaendigt, so dass alle
   * Uebergaenge defininiert sind, es wird dazu ein neuer
   * interner Hilfszustand ergaenzt.
   * @return dieser Automat vervollstaendigt
   */
  public EndlicherAutomat vervollstaendigen() {
    this.vervollstaendigen(Zustand.neu());
    return this;
  }
  
  /** prueft ob der Automat deterministisch ist
   * 
   * @return ist Automat deterministisch?
   */
  public boolean istDeterministisch() {
    if (!this.istOhneEpsilon()) {
      return false;
    }
    for(Terminal ze: this.alphabet) {
      for(Zustand zu: this.zustaende) {
        if (!this.ueber.istEindeutig(zu, ze)) {
          return false;
        }
      }
    }
    return true;
  }
  
  /** prueft ob der Automat vollstaendig definiert ist
   * 
   * @return ist Automat vollstaendig definiert?
   */
  public boolean istVollstaendig() {
    for(Terminal ze: this.alphabet) {
      for(Zustand zu: this.zustaende) {
        if (!this.ueber.istDefiniert(zu, ze)) {
          return false;
        }
      }
    }
    return true;
  }
  
  /** prueft ob der Automat keine Epsilon-Uebergaenge hat
   * 
   * @return hat Automat keine Epsilon-Uebergaenge?
   */
  public boolean istOhneEpsilon() {
    return this.ueber.istOhneEpsilon();
  }
  
  /** Wandelt den Automaten in einen regulaeren Ausdruck um.
   * 
   * @return aus Automat konstruierter regulaerer Ausdruck
   */
  public RegulaererAusdruck alsRegulaererAusdruck() {
    this.deterministisch().minimieren();
    Map<Paar<Zustand, RegulaererAusdruck>, Zustand> tmp = new HashMap<>();
    for(Paar<Zustand,Terminal> pa: this.ueber.getFunktion().keySet()) {
      tmp.put(new Paar<>(pa.getLinks(), pa.getRechts())
          , this.ueber.getFunktion().get(pa).get(0));  // Automat ist deterministisch
    }
    List<Zustand> nichtAnfangOderEnde = new ArrayList<>();
    this.zustaende.forEach(z -> {
      if (!this.endzustaende.contains(z) && !z.equals(this.start)) {
        nichtAnfangOderEnde.add(z);
      }
    });
    while (!nichtAnfangOderEnde.isEmpty()) {
      Zustand weg = nichtAnfangOderEnde.remove(0);
      this.zustandEntfernen(tmp, weg);
    }
       
    RegulaererAusdruck erg = null;
    for (Zustand z: this.endzustaende) {
      Map<Paar<Zustand, RegulaererAusdruck>, Zustand> lokal = this.flacheKopie(tmp);
      for (Zustand weg: this.endzustaende) {
        if(!weg.equals(z) && !weg.equals(this.start)) {
          this.zustandEntfernen(lokal, weg);
        }
      }
      if(z.equals(this.start)) { // Start gleich Ende
        RegulaererAusdruck schleife = null;
        for(Paar<Zustand,RegulaererAusdruck> pa: lokal.keySet()) {
          if (pa.getLinks().equals(z)) {
            if (schleife == null) {
              schleife = pa.getRechts();
            } else {
              schleife = new Oder(schleife, pa.getRechts());
            }
          }
        }
        if(schleife == null) {
          schleife = new Leer();
        }
        if (erg == null) {
          erg = new Stern(schleife);
        } else {
          erg = new Oder(erg, new Stern(schleife));
        } 
      } else { // gibt Zustand z (Endzustand) und Startzustand
        RegulaererAusdruck schleifeStart = null;
        RegulaererAusdruck schleifeEnde = null;
        RegulaererAusdruck startEnde = null;
        RegulaererAusdruck endeStart = null;
        for(Paar<Zustand,RegulaererAusdruck> pa: lokal.keySet()) {
          if (pa.getLinks().equals(z) && lokal.get(pa).equals(z)) {
            if (schleifeEnde == null) {
              schleifeEnde = pa.getRechts();
            } else {
              schleifeEnde = new Oder(schleifeEnde, pa.getRechts());
            }
          }
          if (pa.getLinks().equals(this.start) && lokal.get(pa).equals(this.start)) {
            if (schleifeStart == null) {
              schleifeStart = pa.getRechts();
            } else {
              schleifeStart = new Oder(schleifeStart, pa.getRechts());
            }
          }
          if (pa.getLinks().equals(this.start) && lokal.get(pa).equals(z)) {
            if (startEnde == null) {
              startEnde = pa.getRechts();
            } else {
              startEnde = new Oder(startEnde, pa.getRechts());
            }
          }
          if (pa.getLinks().equals(z) && lokal.get(pa).equals(this.start)) {
            if (endeStart == null) {
              endeStart = pa.getRechts();
            } else {
              endeStart = new Oder(endeStart, pa.getRechts());
            }
          }
        }
          
        RegulaererAusdruck neu = null;
        if (startEnde != null) {
          if (schleifeStart != null && schleifeEnde != null && endeStart != null) {
            // System.out.println("r0");
            neu = new Punkt(
                new Stern(
                    new Oder(schleifeStart, 
                        new Punkt(startEnde, new Stern(schleifeEnde), endeStart)))
                , startEnde
                , new Stern(schleifeEnde)); 
          }
          if (schleifeStart != null && schleifeEnde != null && endeStart == null) {
            // System.out.println("r1");
            neu = new Punkt(
                  new Stern(schleifeStart)
                , startEnde
                , new Stern(schleifeEnde)); 
          }
          if (schleifeStart != null && schleifeEnde == null && endeStart != null) {
            // System.out.println("r2");
            neu = new Punkt(
                new Stern(
                    new Oder(schleifeStart, 
                        new Punkt(startEnde, endeStart)))
                , startEnde); 
          }
          if (schleifeStart != null && schleifeEnde == null && endeStart == null) {
            // System.out.println("r3");
            neu = new Punkt(
                new Stern(schleifeStart)
                , startEnde); 
          }
          if (schleifeStart == null && schleifeEnde != null && endeStart != null) {
            // System.out.println("r4");
            neu = new Punkt(
                new Stern(new Klammern(
                    new Punkt(startEnde, new Stern(schleifeEnde), endeStart)))
                , startEnde
                , new Stern(schleifeEnde)); 
          }
          if (schleifeStart == null && schleifeEnde != null && endeStart == null) {
            // System.out.println("r5");
            neu = new Punkt(
                startEnde
                , new Stern(schleifeEnde)); 
          }
          if (schleifeStart == null && schleifeEnde == null && endeStart != null) {
            // System.out.println("r6");
            neu = new Punkt(
                new Stern(new Klammern(
                   new Punkt(startEnde, endeStart)))
                , startEnde); 
          }
          if (schleifeStart == null && schleifeEnde == null && endeStart == null) {
            // System.out.println("r7");
            neu = startEnde; 
          }
          
        }
        
        if (erg == null) {
          erg = neu;
        } else {
          erg = new Oder(erg, neu);
        }
        
      }
    }
    
    if (erg == null) {
      return new Leer();
    }
    return erg;
  }
  
  protected Map<Paar<Zustand, RegulaererAusdruck>, Zustand> flacheKopie (
      Map<Paar<Zustand, RegulaererAusdruck>, Zustand> map){
    Map<Paar<Zustand, RegulaererAusdruck>, Zustand> erg = new HashMap<>();
    for(Paar<Zustand,RegulaererAusdruck> pa: map.keySet()) {
      erg.put(pa, map.get(pa));
    }
    return erg;
  }

  protected void zustandEntfernen(
      Map<Paar<Zustand, RegulaererAusdruck>, Zustand> tmp,
      Zustand weg) {
    RegulaererAusdruck loop = null; // null; hier fuer keine Loop in weg
    // loop-Berechnung
    for(Paar<Zustand,RegulaererAusdruck> pa: tmp.keySet()) {
      Zustand von = pa.getLinks();
      Zustand nach = tmp.get(pa);
      if (von.equals(weg) && nach.equals(weg)) {
        if (loop == null) {
          loop = pa.getRechts();
        } else {
          loop = new Oder(loop, pa.getRechts());
        }
      }
    }
    if(loop != null) {
      loop = new Stern(loop);
    }
        
    for (Zustand zu1: this.zustaende) {
      for (Zustand zu2: this.zustaende) {
        RegulaererAusdruck vonreg = null; // null; hier fuer keine von --> weg 
        RegulaererAusdruck nachreg = null; // null; hier fuer keine weg --> nach    
        for(Paar<Zustand,RegulaererAusdruck> pa: tmp.keySet()) {
          Zustand von = pa.getLinks();
          Zustand nach = tmp.get(pa);
          if (von.equals(zu1) && nach.equals(weg) && !zu1.equals(weg)) {
            if (vonreg == null) {
              vonreg = pa.getRechts();
            } else {
              vonreg = new Oder(vonreg, pa.getRechts());
            }
          }
        }
        
        for(Paar<Zustand,RegulaererAusdruck> pa: tmp.keySet()) {
          Zustand von = pa.getLinks();
          Zustand nach = tmp.get(pa);
          if (von.equals(weg) && nach.equals(zu2) && !zu2.equals(weg)) {
            if (nachreg == null) {
              nachreg = pa.getRechts();
            } else {
              nachreg = new Oder(nachreg, pa.getRechts());
            }
          }
        }
        
        if(vonreg != null && nachreg != null) {
          if(loop != null ) {
            tmp.put(new Paar<>(zu1, new Punkt(vonreg, loop, nachreg)), zu2);
          } else {
            tmp.put(new Paar<>(zu1, new Punkt(vonreg, nachreg)), zu2);
          }          
        }
      }
    }
    
    // loesche Knoten weg
    List<Paar<Zustand,RegulaererAusdruck>> loeschen = new ArrayList<>();
    for(Paar<Zustand,RegulaererAusdruck> pa: tmp.keySet()) {
      Zustand erg = tmp.get(pa);
      if (pa.getLinks().equals(weg) || erg.equals(weg)) {
        loeschen.add(pa);
      }
    }
    for(Paar<Zustand,RegulaererAusdruck> pa: loeschen) {
      tmp.remove(pa);
    }
  }

  /** Entfernt bei einem deterministischen Automaten alle Zustaende,
   * die nicht erreichbar sind.
   * @return this mit ausschliesslich erreichbaren Zustaenden
   */
  public EndlicherAutomat entferneUnerreichbareZustaende() {
    if(! this.istDeterministisch()) {
      throw new IllegalStateException("die Entfernung unerreichbarer"
          + " Zustaende nur fuer"
          + " deterministische Automaten implementiert:\n" + this);
    }
    Map<Zustand, Wort> erreichen = this.zustandErreichbarMit();
    List<Zustand> zuloeschen = new ArrayList<>();
    for(Zustand z: this.zustaende) {
      if (erreichen.get(z) == null) {
        zuloeschen.add(z);
      }
    }
    for(Zustand z: zuloeschen) {
      this.zustaende.remove(z);
      this.endzustaende.remove(z);
      for(Zustand ende: this.zustaende) {
        for(Terminal t: this.alphabet) {
          this.ueber.remove(z, t, ende);
        }
      }
    }
    return this;
  }
  
  /** berechnet zu diesem Automaten sprachaequivalenten Automaten
   * mit minimaler Anzahl an Zustaenden, der Automat muss dazu 
   * deterministisch sein
   * @return aequivalenter minimaler Automat
   */
  public EndlicherAutomat minimieren() {
    if(! this.istDeterministisch()) {
      throw new IllegalStateException("die Minimierung klappt nur fuer"
          + " deterministische Automaten:\n" + this);
    }
    this.vervollstaendigen();
    this.entferneUnerreichbareZustaende();
    this.minimierungzustaende = new ArrayList<>();
    for(Zustand zu: this.zustaende) {
      this.minimierungzustaende.add(zu);
    }
    EndlicherAutomat erg = new EndlicherAutomat();
    erg.alphabet = this.alphabet;

    int zustaende = this.zustaende.size();
    Map<Zustand, Integer> zustandAlsNummer = new HashMap<>();
    for(int i = 0; i < zustaende; i++) {
      zustandAlsNummer.put(this.zustaende.get(i), i);
    }
    this.aequivalent = new boolean[zustaende][zustaende];
    for(int i = 0; i < zustaende - 1; i++) {
      for (int j = i + 1; j < zustaende; j++) {
        this.aequivalent[i][j] = true;
      }
    }
    //this.zeigBooleanArray(zustaende, aequivalent);
    
    for(int i = 0; i < zustaende - 1; i++) {
      for (int j = i + 1; j < zustaende; j++) {
        if (this.endzustaende.contains(this.zustaende.get(i))
                && !this.endzustaende.contains(this.zustaende.get(j))
            || this.endzustaende.contains(this.zustaende.get(j)) 
                && !this.endzustaende.contains(this.zustaende.get(i))) {
          this.aequivalent[i][j] = false;
        }
      }
      //this.zeigBooleanArray(zustaende, aequivalent);
    }
    
    //this.zeigBooleanArray(zustaende, aequivalent);
    
    boolean neuesGefunden = true;
    //int zaehler = 0;
    while(neuesGefunden) {
    //while(zaehler < zustaende) {
      neuesGefunden = false;
      for(int i = 0; i < zustaende - 1; i++) {
        for (int j = i + 1; j < zustaende; j++) {
          if (this.aequivalent[i][j]) {
            for(Terminal z: this.alphabet) {
              Zustand zi = this.ueber.moeglicheFolgezustaende(this.zustaende.get(i), z).get(0);
              Zustand zj = this.ueber.moeglicheFolgezustaende(this.zustaende.get(j), z).get(0);
//              if (this.endzustaende.contains(zi) && !this.endzustaende.contains(zj)
//                  || this.endzustaende.contains(zj) && !this.endzustaende.contains(zi)) {
              // beide Seiten pruefen, da eine Haelfte der Matrix immer false
              // für gleiche Zustaende ist der Eintrag immer false, deshalb vermeiden
              if (!this.aequivalent[zustandAlsNummer.get(zi)][zustandAlsNummer.get(zj)]
                     && !this.aequivalent[zustandAlsNummer.get(zj)][zustandAlsNummer.get(zi)]
                     && !zi.equals(zj) ) {
                this.aequivalent[i][j] = false;
                neuesGefunden = true;
              }
            }
          }
        }
        //this.zeigBooleanArray(zustaende, aequivalent);
      }
      //zaehler ++;
    }
    //this.zeigBooleanArray(zustaende, aequivalent);
    
    List<Zustand> abgearbeitet = new ArrayList<>();
    List<HashSet<Zustand>> neueZustaende = new ArrayList<>();
    Map<Zustand, HashSet<Zustand>> zugeordneteKlasse = new HashMap<>();
    //this.zustaende.forEach(z -> abgearbeitet.add(z));
    for(int i = 0; i < zustaende; i++) {
      Zustand zst = this.zustaende.get(i);
      if (!abgearbeitet.contains(zst)) {
        HashSet<Zustand> neu = new HashSet<>();
        neu.add(zst);
        zugeordneteKlasse.put(zst, neu);
        for (int j = i + 1; j < zustaende; j++) {
          if(aequivalent[i][j]) {
            neu.add(this.zustaende.get(j));
            abgearbeitet.add(this.zustaende.get(j));
            zugeordneteKlasse.put(this.zustaende.get(j), neu);
          }
        }
        neueZustaende.add(neu);
      }
    }
    
    Map<HashSet<Zustand>, Zustand> namen = this.zustandsmengenInZustaende(
        erg, neueZustaende);
    
    erg.ueber = new AutomatUeberfuehrungsfunktion();
    erg.start = namen.get(zugeordneteKlasse.get(this.start));
    //System.out.println("zugeordnet: " + zugeordneteKlasse + " " + zustaende + " " + namen + " " + neueZustaende);
    
    List<Zustand> bearbeitet = new ArrayList<>();
    for(Zustand zu: this.zustaende) {
      if(! bearbeitet.contains(zu)) {
        HashSet<Zustand> klasse = zugeordneteKlasse.get(zu);
        for(Terminal ze: this.alphabet) {
          erg.ueber.add(namen.get(klasse)
              , ze
              , namen.get(zugeordneteKlasse.get(this.ueber.moeglicheFolgezustaende(zu, ze).get(0))));
        }
        bearbeitet.addAll(klasse);
      }
    }
    
    this.alphabet = erg.alphabet;
    this.endzustaende = erg.endzustaende;
    this.start = erg.start;
    this.ueber = erg.ueber;
    this.zustaende = erg.zustaende;
    return this;
  }

  /** Gibt nach dem Aufruf von minimieren() die berechnete
   *  Minimierungsmatrix in der Konsole auf. 
   */
  public void zeigeMinimierungsmatrix() { 
    int zeichenanzahl = 6; // angezeigte Zeichen des Zustandsnamens
    if (this.aequivalent == null 
     //   || this.aequivalent.length != this.zustaende.size()
        ){
      System.err.println("Minimierungsmatrix nur sinnvoll direkt nach"
          + " einer Minimierung anzeigbar.");
      return;
    }
    int zustaende = this.minimierungzustaende.size();
    String[] namen = new String[zustaende];
    for(int i = 0; i < zustaende; i++) {
      String tmp = this.minimierungzustaende.get(i).toString();
      while(tmp.length() < zeichenanzahl) {
        tmp = " " + tmp;
      }
      namen[i] = tmp.substring(0, zeichenanzahl);
    }
    for (int j = 0; j < zeichenanzahl; j++) {
      for(int i = 0; i < zustaende; i++) {
        System.out.print(" " + namen[i].substring(j, j+1));
      }
      System.out.println("");
    }
    
    //System.out.println("");
    for(int i = 0; i < zustaende - 1; i++) {
      for (int j = 0; j < i + 1; j++) {
        System.out.print("  ");
      }
      for (int j = i + 1; j < zustaende; j++) {
        System.out.print(" " + (aequivalent[i][j]?" ":"X"));
      }
      System.out.println("  " + this.minimierungzustaende.get(i));
    }
    System.out.println("---------------------------------");
  }
  
  /** wandelt diesen Automaten in sprachaequivalenten deterministischen
   * Automaten um
   * @return dieser Automat in deterministischer Form
   */
  public EndlicherAutomat deterministisch() {
    return this.deterministisch(true);
  }
  
  /** wandelt diesen Automaten in sprachaequivalenten deterministischen
   *  Automaten um
  * @param vervollstaendigen soll Automat vorher vervollstaendigt 
  *        werden
  * @return dieser Automat in deterministischer Form
  */
  public EndlicherAutomat deterministisch(boolean vervollstaendigen) {
    EndlicherAutomat erg = new EndlicherAutomat();
    if(!this.istOhneEpsilon()) {
      this.epsilonEntfernen();
      //System.out.println("eps entfernen");
    }
    if (vervollstaendigen) {
      this.vervollstaendigen();
    }
    if(this.istDeterministisch()) {
      return this;
    }
    // HashSet statt Set, da Paar kein Set mag (nicht Cloneable)
    List<HashSet<Zustand>> abzuarbeiten = new ArrayList<>();
    List<HashSet<Zustand>> abgearbeitet = new ArrayList<>();
    HashSet<Zustand> start = new HashSet<>();
    start.add(this.start);
    abzuarbeiten.add(start);
    
    Map<Paar<HashSet<Zustand>, Terminal>, HashSet<Zustand>> ueber = new HashMap<>();
    while (!abzuarbeiten.isEmpty()) {
      HashSet<Zustand> aktuell = abzuarbeiten.remove(0);
      for(Terminal ze: this.alphabet) {
        HashSet<Zustand> next = new HashSet<>();
        for(Zustand zu: aktuell) {
          List<Zustand> nzst = new ArrayList<>(); 
          try {
            nzst = this.ueber.moeglicheFolgezustaende(zu, ze);
          } catch(Exception e) {}
          next.addAll(nzst);
        }
        ueber.put(new Paar<>(aktuell, ze), next);
        if(!abgearbeitet.contains(next) && !abzuarbeiten.contains(next)
            && !aktuell.equals(next)) {
          abzuarbeiten.add(next);
        }
      }
      abgearbeitet.add(aktuell);
    }
    
    erg.zustaende = new ArrayList<>();
    erg.endzustaende = new ArrayList<>();
    
    //wandle Zustands-Sets in Strings um
    Map<HashSet<Zustand>, Zustand> namen = this.zustandsmengenInZustaende(
        erg, abgearbeitet);
    
    erg.alphabet = this.alphabet;
    erg.start = namen.get(start);
    erg.ueber = new AutomatUeberfuehrungsfunktion();
    
    //Ueberfuehrungsfunktion uebertragen
    //Map<Paar<HashSet<Zustand>, Terminal>, HashSet<Zustand>> ueber
    ueber.forEach((paar, zustaende) -> {
      erg.ueber.add(namen.get(paar.getLinks()), paar.getRechts(), namen.get(zustaende));
    });
    
    this.alphabet = erg.alphabet;
    this.endzustaende = erg.endzustaende;
    this.start = erg.start;
    this.ueber = erg.ueber;
    this.zustaende = erg.zustaende;
    
    return this;
  }
  
  /**
   * Methode wandelt die als zweiten Parameter uebergebenen Zustaende
   * in neue einfache Zustaende um, deren Namen sich aus den Namen der
   * einzelnen Zustaende getrennt mit einem _ zusammensetzt, die neuen
   * Zustaende und die dadurch gefundenen Endzustaende (wenn einer der
   * Zustaende als Ende markiert ist) werden im als ersten Parameter
   * uebergebenen Automaten hinzugefuegt
   * @param auto Automat, dem die neuen Zustaende und Endzustaende
   *            hinuigefuegt werden sollen
   * @param zustandsmengen Liste von Zustandsmengen, die jeweils in
   *                     einzelne Zustaende umgewandelt werden
   * @return Zuordnung von Zustandsmengen zum neu zugeordneten Zustand
   */
  protected Map<HashSet<Zustand>, Zustand> zustandsmengenInZustaende(
      EndlicherAutomat auto, List<HashSet<Zustand>> zustandsmengen) {
    Map<HashSet<Zustand>, Zustand> namen = new HashMap<>();
    for(HashSet<Zustand> set: zustandsmengen) {
      StringBuilder sb = new StringBuilder();
      boolean ende = false; // enthaelt Endzustand?
      for(Zustand zu: this.zustaende) {
        if (set.contains(zu)) {
          sb.append(zu.getName() + "_"); // _ damit Namen eindeutig bleiben
          if (!ende && this.endzustaende.contains(zu)) {
            ende = true;
          }
        }
      }
      if (sb.length() != 0) {
        sb.deleteCharAt(sb.length() - 1);
        while(Zustand.istZustand(sb.toString())) {
          sb.append("_");
        }
      } else {
        sb.append("_");
      }
      Zustand neu = Zustand.istZustand(sb.toString())
          ? Zustand.zustand(sb.toString()) 
              : Zustand.zustand(sb.toString());
      auto.zustaende.add(neu);
      if(ende) {
        auto.endzustaende.add(neu);
/////        neu.setEnde(true);
      }
      namen.put(set, neu);
    }
    return namen;
  }
  
  /** Wandelt Automaten in eine rechtslineare Grammatik um, die die
   * gleiche Sprache erzeugt, die der Automat akzeptiert.
   * @return sprachaequivalente Grammatik
   */
  public KontextfreieGrammatik alsGrammatik() {
    String[] praefix={"_", "#", "$", "§", "@", "€", "³"}; // soll Nichtterminale und Zustandsnamen praefixfrei machen
    KontextfreieGrammatik erg = new KontextfreieGrammatik();
    Map<String, Nichtterminal> zustaende = new HashMap<>();
    
    this.zustaende.forEach(z -> {
      Nichtterminal nerg = null;
      String name = z.getName();
      int i = 0;
      while(nerg == null) { // suche praefixfreienNamen
        try {
          nerg = Nichtterminal.nichtterminal(name);
        } catch(IllegalStateException e) {
          name = praefix[i%praefix.length] + name;
        }
      }    
      erg.addNichtterminale(nerg);
      zustaende.put(z.getName(), nerg);
    });
    this.alphabet.forEach(z -> {
      erg.addTerminale(Terminal.terminal(z.getZeichen()));
    });
    erg.setStart(zustaende.get(this.start.getName()));
    for(Zustand zu: this.zustaende) {
      for(Terminal ze: this.alphabet) {
        String name = ze.getZeichen();
        try {
          List<Zustand> next = this.ueber.moeglicheFolgezustaende(zu, ze);
          for(Zustand ne: next) {
            erg.addRegeln(new KontextfreieRegel(
                zustaende.get(zu.getName())
                , new Wort(Terminal.terminal(name)
                    , zustaende.get(ne.getName()))));
          }
        } catch (Exception e) {
          // System.out.println(zu + " " + ze + " " + e);
        }
      }
      List<Zustand> wolke = this.ueber.mitEpsilonErreichbar(zu);
      for(Zustand wo:wolke) {
        if (this.endzustaende.contains(wo)) {
          erg.addRegeln(new KontextfreieRegel(
              zustaende.get(zu.getName()), new Wort()));
        }
      }
    }
    
    return erg;
  }
  
  public KontextfreieGrammatik alsGrammatikVers2() {
    String prefix = "#";
    KontextfreieGrammatik erg = new KontextfreieGrammatik();
    this.zustaende.forEach(z -> {
      String name = prefix + z.getName();
      if(! Nichtterminal.istZeichen(name)) {
        erg.addNichtterminale(Nichtterminal.nichtterminal(name));
      }
    });
    this.alphabet.forEach(z -> {
      String name = "\u001b[1m" + z.getZeichen() + "\u001b[0m";
      if(! Terminal.istZeichen(name)) {
        erg.addTerminale(Terminal.terminal(name));
      }
    });
    erg.setStart(Nichtterminal.nichtterminal(prefix + this.start.getName()));
    for(Zustand zu: this.zustaende) {
      for(Terminal ze: this.alphabet) {
        String name = "\u001b[1m" + ze.getZeichen() + "\u001b[0m";
        try {
          List<Zustand> next = this.ueber.moeglicheFolgezustaende(zu, ze);
          for(Zustand ne: next) {
            erg.addRegeln(new KontextfreieRegel(
                Nichtterminal.nichtterminal(prefix + zu.getName())
                , new Wort(Terminal.terminal(name)
                    , Nichtterminal.nichtterminal(prefix + ne.getName()))));
          }
        } catch (Exception e) {
          // System.out.println(zu + " " + ze + " " + e);
        }
      }
      if (this.endzustaende.contains(zu)) {
        erg.addRegeln(new KontextfreieRegel(
            Nichtterminal.nichtterminal(prefix + zu.getName()), new Wort()));
      }
    }
    
    return erg;
  }
  
  
  /** Der Automat wird so vervollstaendigt, dass für jeden Zustand
   * und jedes Zeichen ein Übergang definiert ist; dies macht nur nach
   * der Entferung von Epsilon-Uebergaengen Sinn; der uebergene Zustand
   * wird als absorbierender Zustand ergaenzt, auf den fehlende
   * Uebergaenge zulaufen und der nicht verlassen werden kann.
   * Sollte es Epsilon-Uebergaenge geben, wird eine 
   * IllegalStateException geworfen.
   * @param dummy absorbierender Zustand
   * @return dieser Automat (this) vervollstaendigt
   */
  public EndlicherAutomat vervollstaendigen(Zustand dummy) {
    if (this.istVollstaendig()) {
      return this;
    }
    if(!this.istOhneEpsilon()) {
      throw new IllegalStateException("Vervollstaendigen vor der "
          + "Entfernung von Epsilon-Uebergaengen macht keinen Sinn.");
    }
    if (!this.zustaende.contains(dummy)) {
      this.zustaende.add(dummy);
    }
    for(Terminal ze: this.alphabet) {
      for(Zustand zu: this.zustaende) {
        if(! this.ueber.istDefiniert(zu, ze)) {
          this.ueber.add(zu, ze, dummy);
        }
      }
      this.ueber.add(dummy, ze, dummy);
    }
    return this;
  }
  
  /** Die existierenden Zustaende werden 1:1 gegen neue generierte
   * Zustaende ersetzt, da bei Berechnungen (deterministisch, minimal)
   * sehr lange Zustandsnamen entstehen koennen.
   * @return dieser Automat (this) mit ersetzten Zustaenden
   */
  public EndlicherAutomat zustandsnamenKuerzen() {
    EndlicherAutomat ea = new EndlicherAutomat();
    Map<Zustand,Zustand> umbenennung = new HashMap<>();
    for(Zustand z: this.zustaende) {
      Zustand neu = Zustand.neu();
      umbenennung.put(z, neu);
      ea.zustaende.add(neu);
      if (this.endzustaende.contains(z)) {
        ea.endzustaende.add(neu);
      }
    }
    for(Paar<Zustand, Terminal> pa: this.ueber.getFunktion().keySet()) {
      List<Zustand> nach = this.ueber.getFunktion().get(pa);
      List<Zustand> neu = new ArrayList<>();
      nach.forEach(z -> neu.add(umbenennung.get(z)));
      Paar<Zustand, Terminal> paneu 
          = new Paar<>(umbenennung.get(pa.getLinks()), pa.getRechts());
      ea.ueber.getFunktion().put(paneu, neu);
    }
    for(Zustand pa: this.ueber.getEpsilon().keySet()) {
      List<Zustand> nach = this.ueber.getEpsilon().get(pa);
      List<Zustand> neu = new ArrayList<>();
      nach.forEach(z -> neu.add(umbenennung.get(z)));
      ea.ueber.getEpsilon().put(umbenennung.get(pa), neu);
    }
    ea.start = umbenennung.get(this.start);
    
    //this.alphabet = ea.alphabet;
    this.endzustaende = ea.endzustaende;
    this.start = ea.start;
    this.ueber = ea.ueber;
    this.zustaende = ea.zustaende;
    return this;
  }

  /** Methode dient zur Pruefung ob zwei Automaten 
   * sprachaequivalent sind. Sollten die Automaten ein
   * unterschiedliches Alphabet nutzen, 
   * wird ein eine IllegalStateException geworfen.
   * @param ea zu vergleichender Automat
   * @return this und ea isomorph?
   */
  public boolean istSprachaequivalent(EndlicherAutomat ea) {
    return this.clone().deterministisch().minimieren()
        .istMinimalIsomorph(ea.clone().deterministisch().minimieren());
  }
  
  /** Methode dient zur Pruefung ob zwei minimale Automaten
   * isomorph (deckungsgleich) sind. Sollten die Automaten ein
   * unterschiedliches Alphabet nutzen oder nichtdeterministisch
   * sein, wird ein eine IllegalStateException geworfen.
   * @param ea zu vergleichender Automat
   * @return this und ea isomorph?
   */
  public boolean istMinimalIsomorph(EndlicherAutomat ea) {
    System.out.println(this + "\n" + ea);
    
    if(!this.istDeterministisch() || !ea.istDeterministisch()) {
      throw new IllegalStateException("nur fuer deterministische"
          + " Automaten umgesetzt");
    }
    for(Terminal z: this.getAlphabet()) {
      if (!ea.getAlphabet().contains(z)) {
        //System.out.println(z + " nicht in\n" + ea);
        throw new IllegalStateException("gleiches Alphabet notwendig,"
            + " uebergebenem Automaten fehlt " + z + "\n" 
            + ea.getAlphabet().get(0) + "  " 
            + ea.getAlphabet().get(0).getClass() + "\n" + ea);
      }
    }
    for(Terminal z: ea.getAlphabet()) {
      if (!this.getAlphabet().contains(z)) {
        throw new IllegalStateException("gleiches Alphabet notwendig,"
            + " this-Automaten fehlt " + z + "\n");
      }
    }
    if(this.zustaende.size() != ea.zustaende.size()) {
      return false;
    }
    if(this.endzustaende.contains(this.start) 
            && !ea.getEndzustaende().contains(ea.getStart())
        || !this.endzustaende.contains(this.start) 
            && ea.getEndzustaende().contains(ea.getStart())) {
      System.err.println("Leeres wort wird von einem Automaten "
          + "akzeptiert, vom anderen nicht.");
      return false;
    }
    Map<Zustand, Zustand> eq = new HashMap<>(); // gleich erkannte Zustaende
    //Map<Paar<Zustand, Zustand>, Wort> wort = new HashMap<>(); // Beispielwort, mit dem Aequivalenz berechnet wurde
    eq.put(this.start, ea.getStart());
    //wort.put(new Paar<>(this.start, ea.getStart()), new Wort(""));
    int anzahl = 0;
    while(anzahl < eq.size()) {
      anzahl = eq.size();
      for(Terminal ze: this.alphabet) {
        //Iterator<Zustand> it = eq.keySet().iterator();
        List<Paar<Zustand, Zustand>> hinzu = new ArrayList<>();
        for(Zustand zu: eq.keySet()) {
          Zustand neu = this.getUeber()
              .getFunktion()
              .get(new Paar<Zustand, Terminal>(zu, ze))
              .get(0);
          Zustand neu2 = ea.getUeber()
              .getFunktion()
              .get(new Paar<Zustand, Terminal>(eq.get(zu), ze))
              .get(0);
          Zustand pot = eq.get(neu);
          if (pot == null) {
            if(this.endzustaende.contains(neu) && !ea.getEndzustaende().contains(neu2)) {
      //        System.err.println(neu2 + " muesste Endzustand sein: "
      //            + wort.get(new Paar<>(this.start, ea.getStart())).anfuegen(ze));
              return false;
            }
            if(!this.endzustaende.contains(neu) && ea.getEndzustaende().contains(neu2)) {
      //        System.err.println(neu2 + " darf kein Endzustand sein: "
      //            + wort.get(new Paar<>(this.start, ea.getStart())).anfuegen(ze));
              return false;
            }
            //eq.put(neu, neu2);
            hinzu.add(new Paar<>(neu, neu2));
      //      wort.put(new Paar<>(neu, neu2)
      //          , wort.get(new Paar<>(this.start, ea.getStart())).anfuegen(ze));
          } else {
            if (!pot.equals(neu2)) {
              System.err.println(pot + " muesste aequivalent zu " + neu2 
                  + " sein");
              return false;
            }
          }
        }
        for(Paar<Zustand, Zustand> p: hinzu) {
          eq.put(p.getLinks(), p.getRechts());
        }
      }  
    }
    return true;
  }
  
  /** Findet fuer zwei Automaten mit unterschiedlicher Sprache
   * und gleichem Alphabet (wird nicht geprueft) 
   * ein Wort, das nur in einer der Sprachen enthalten ist.
   * Sollten Automaten die gleiche Sprache akzeptieren, wird
   * eine IllegalStateException geworfen
   * @param ea2 zu vergleichender Automat
   * @return nicht gemeinsames Wort
   */
  public Wort nichtGemeinsamesWort(EndlicherAutomat ea2) {
  EndlicherAutomat thiss = this.clone().deterministisch();    
  EndlicherAutomat ea = ea2.clone().deterministisch();  
    if(thiss.endzustaende.contains(thiss.start) != 
           ea.getEndzustaende().contains(ea.getStart())) {
      return new Wort();
    }
    
    Map<Wort,Paar<Zustand, Zustand> > abarbeitung = new HashMap<>();
    abarbeitung.put(new Wort(""), new Paar<>(thiss.start, ea.start));
    int zeichenanzahl = 0;
    
    while (zeichenanzahl <= thiss.zustaende.size() || zeichenanzahl <= ea.zustaende.size()) {
      zeichenanzahl ++;
      Map<Wort,Paar<Zustand, Zustand> > tmp = new HashMap<>();
      for (Wort w: abarbeitung.keySet()) {
        Paar<Zustand, Zustand> p = abarbeitung.get(w);
        for(Terminal z: thiss.alphabet){
          Wort neu = w.clone().anfuegen(z);
          Zustand links = thiss.ueber.moeglicheFolgezustaende(p.getLinks(), z).get(0);
          Zustand rechts = ea.ueber.moeglicheFolgezustaende(p.getRechts(), z).get(0);
          if(thiss.endzustaende.contains(links) != ea.endzustaende.contains(rechts)) {
            return neu;
          }
          tmp.put(neu, new Paar<>(links, rechts));
        };
      };
      abarbeitung = tmp;
    }
    throw new IllegalStateException("beide Automaten akzeptieren"
            + " die gleiche Sprache.");
  }  
  
  /** Berechnet zu jedem Zustand eines der kuerzesten Worte um
   * diesen zuerreichen
   * @return Zurordnung von Zustaenden und Woertern, die zu
   *         dem jeweiligen Zustand fuehren
   */
  public Map<Zustand, Wort> zustandErreichbarMit() {
    List<Zustand> besucht = new ArrayList<>();
    Map<Zustand, Wort> wort = new HashMap<>();
    besucht.add(this.start);
    wort.put(this.start, new Wort());
    int anzahl = 0;
    List<Zustand> neu = new ArrayList<>(); // im naechsten Schritt abzuarbeiten
    neu.add(this.start);
    while(anzahl < besucht.size()) {
      anzahl = besucht.size();
      List<Zustand> nextneu = new ArrayList<>();
      for(Terminal ze: this.alphabet) {
        for(Zustand ne:neu) {
          Zustand tmp = this.getUeber()
              .getFunktion()
              .get(new Paar<Zustand, Terminal>(ne, ze))
              .get(0);
          if(!besucht.contains(tmp)) {
            besucht.add(tmp);
            wort.put(tmp, wort.get(ne).clone().anfuegen(ze));
            nextneu.add(tmp);
          }
        }
      }
      neu = nextneu;
      //System.out.println("besucht: " + besucht + " " + wort);
    }
    return wort;
  }
  
  
  
  
  public EndlicherAutomat addUeberfuehrung(Zustand alt, Terminal lesen
      , Zustand neu) { 
    this.ueber.add(alt, lesen, neu);
    return this;
  }
  
  public EndlicherAutomat addUeberfuehrung(String alt, String lesen
      , String neu) { 
    this.ueber.add(this.zustandFallsExistert(alt)
        , Terminal.terminal(lesen)
        , this.zustandFallsExistert(neu));   
    return this;
  }
  
  public EndlicherAutomat addEpsilonueberfuehrung(String alt
      , String neu) { 
    this.ueber.addEpsilon(this.zustandFallsExistert(alt)
        , this.zustandFallsExistert(neu));
    return this;
  }
  
  public EndlicherAutomat addEpsilonueberfuehrung(Zustand alt
      , Zustand neu) { 
    this.ueber.addEpsilon(alt, neu);
    return this;
  }
  
  public EndlicherAutomat addZustand(String... namen) {
    for(String s:namen) {
      this.zustaende.add(Zustand.zustand(s));
    }
    return this;
  }
  
  public EndlicherAutomat addZustand(Zustand... namen) {
    for(Zustand s:namen) {
      this.zustaende.add(s);
    }
    return this;
  }
  
  public EndlicherAutomat addZeichen(String... namen) {
    for(String s:namen) {
      this.alphabet.add(Terminal.terminal(s));
    }
    return this;
  }
  
  public EndlicherAutomat addZeichen(Terminal... namen) {
    for(Terminal s:namen) {
      this.alphabet.add(s);
    }
    return this;
  }
  
  @Override
  public String toString() {
    return "Zustaende: " + this.zustaende 
        + "\nEndzustaende: " + this.endzustaende
        + "\nAlphabet: " + this.alphabet
        + "\nStart: " + this.start
        + "\n" + this.ueber;
  }
  
  @Override
  public EndlicherAutomat clone() {
    return this.deepClone();
  }

  public List<Zustand> getZustaende() {
    return zustaende;
  }

  public void setZustaende(List<Zustand> zustaende) {
    this.zustaende = zustaende;
  }

  public List<Zustand> getEndzustaende() {
    return endzustaende;
  }

  public void setEndzustaende(List<Zustand> endzustaende) {
    this.endzustaende = endzustaende;
  }

  public List<Terminal> getAlphabet() {
    return alphabet;
  }

  public void setAlphabet(List<Terminal> alphabet) {
    this.alphabet = alphabet;
  }

  public Zustand getStart() {
    return start;
  }

  public void setStart(Zustand start) {
    this.start = start;
  }

  public AutomatUeberfuehrungsfunktion getUeber() {
    return ueber;
  }

  public void setUeber(AutomatUeberfuehrungsfunktion ueber) {
    this.ueber = ueber;
  }

  public AutomatKonfiguration getKonfiguration() {
    return konfiguration;
  }

  public void setKonfiguration(AutomatKonfiguration konfiguration) {
    this.konfiguration = konfiguration;
  }
}
