package grammatik;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import alphabet.Nichtterminal;
import alphabet.Terminal;
import alphabet.Wort;
import alphabet.Zeichen;
import ausfuehrung.Ableitung;
import ausfuehrung.CYKKnoten;
import ausfuehrung.Regelnutzung;
import endlicherAutomat.EndlicherAutomat;
import semantik.Ableitungsbaum;
import util.Eingabe;
import zustand.Zustand;

public class KontextfreieGrammatik implements Cloneable{
	protected Set<Terminal> terminale = new HashSet<>();
	protected Set<Nichtterminal> nichtterminale = new HashSet<>();
	protected List<KontextfreieRegel> regeln = new ArrayList<>(); // keine Map um Reihenfolge zu erhalten
	protected Nichtterminal start;
	protected Map<KontextfreieRegel, List<Regelnutzung>>  ergaenzteRegeln = new HashMap<>();
	protected boolean ignoriereLeerzeichen = false;
	protected HashSet<Nichtterminal>[][] cykTabelle;
	

  public KontextfreieGrammatik() {
  }
	
	/**
	 * Schreibt die aktuelle kontextfreie Grammatik in eine Datei mit
	 * dem gegebenen Namen datei. Das Resultat ist wieder einlesbar.
	 * Sollte es Probleme geben, wird eine Warnung ausgegeben.
	 * @param datei (Pfad und) Name der Zieldatei
	 */
	public void dateiSpeichern(String datei) {
	  File file = new File(datei);
	  try {
      if(!file.exists()) {
        file.createNewFile();
      }
    } catch (IOException e1) {
      throw new IllegalStateException("Datei " + datei 
          + " nicht erzeugbar");
    }
    Path path = Paths.get(datei);
    try (BufferedWriter writer = Files.newBufferedWriter(path, Charset.forName("UTF-8"))) {
      StringBuilder sb = new StringBuilder("N: ");
      for(Nichtterminal z: this.nichtterminale) {
        sb.append(z.toString() + " ");
      }
      writer.write(sb.toString() +"\n");
      StringBuilder sb2 = new StringBuilder("T: ");
      for(Terminal z: this.terminale) {
        if(z.getZeichen().equals("/space")) {
          sb2.append("/space ");
        } else {
          sb2.append(z.toString() + " ");
        }
      }
      writer.write(sb2.toString() + "\n");
      writer.write("S: " + this.start +"\n");
      for(KontextfreieRegel reg: this.regeln) {
        writer.write(reg.toFile() + "\n");
      }
      if (this.ignoriereLeerzeichen) {
        writer.write("%ignorespace\n");
      }
      
    } catch(Exception e) {
      System.err.println("Warnung: Problem beim Schreiben in die Datei " 
          + datei + ": " + e);
      e.printStackTrace();
    }
	}
	
	/**
	 * Liest aus der Datei datei eine kontextfreie Grammatik ein, 
	 * sollte es bereits etwas in this geben, werden diese Informationen
	 * ergaenzt (was auch zu konflikten fuehren kann). Um Probleme zu
	 * vermeiden, sollte das Lesen der erste Schritt nach der 
	 * Objekterzeugung sein
	 * @param datei (Pfad und) Dateiname, aus der Grammatik gelesen wird 
	 */
	 public void dateiEinlesen(String datei) {
	    File file = new File(datei);
	      if (file.exists() && file.canRead()) {
	        Path path = Paths.get(datei);
	        try (BufferedReader reader = Files
	            .newBufferedReader(path, Charset.forName("UTF-8"))) {

	          String currentLine = null;
	          String start = "";
	          List<String> regeln = new ArrayList<>();
	          while ((currentLine = reader.readLine()) != null) {
	            start = verarbeiteZeile(currentLine, start, regeln);         
	          }
	          if("".equals(start)) {
	            throw new IllegalStateException("Grammatik in " + datei 
	                + " enthaelt kein Startsymbol");
	          }
	          this.setStart(Nichtterminal.nichtterminal(start));
	          this.neueRegeln(regeln);
	        } catch(Exception e) {
	          System.err.println("WARNING: " + datei 
	              + " existiert: " + file.exists()
	              + " lesbar: " + file.canRead()
	              + " meldung: " + e
	          );
	        }
	      }else {
          System.err.println("WARNING: " + datei 
              + " existiert: " + file.exists()
              + " lesbar: " + file.canRead()
          );
    }
	  }
	 
	 /** Der uebergebene String wird als Grammatik aufgefasst, Zeilen 
	  * muessen durch den Zeilentrenner /n (auch Mac) getrennts sein,
	  * es muss Terminale (T:), Nichtterminale (N:), ein Startsymbol (S:)
	  * und Regeln geben, die in dieses Objekt eingelesen werden. 
	  * @param gram String, der Grammatik enthaelt, die in this ergaenzt wird
	  */
	 public void stringAlsGrammatik(String gram) {
	   String[] zeilen = gram.split("\\n");
	   String start = "";
     List<String> regeln = new ArrayList<>();
     for(String s:zeilen) {
       //System.out.println(" zeile: " + s);
       start = verarbeiteZeile(s, start, regeln);    
       //System.out.println(" nt: " + this.nichtterminale);
     }
     if("".equals(start)) {
       throw new IllegalStateException("Grammatik in String"
           + " enthaelt kein Startsymbol: " + gram);
     }
     this.setStart(Nichtterminal.nichtterminal(start));
     this.neueRegeln(regeln);
	 }

	 /**
	  * Die Methode verarbeitet eine Grammatik-Zeile currentline. Erkannte
	  * Regeln werden in die uebergebene Liste regeln eingetragen, das
	  * aktuelle Startsymbol kann uebergeben werden, ein aktuell gefundenes
	  * Startsymbol ist das Ergebnis der Methode.
	  * @param currentLine zu verarbeitende Zeile
	  * @param start aktuelles Startsymbol
	  * @param regeln Liste zur Aufnahme von Regeln
	  * @return neues Startsymbol, oder altes, wenn nicht gefunden
	  */
  private String verarbeiteZeile(String currentLine, String start,
      List<String> regeln) {
    //System.out.println("  pl: " + currentLine);
    String trim = currentLine.trim();
    if(trim.length() == 0) {
      return start;
    }
    if(trim.startsWith("%")) {
      String kommentar = trim.substring(1, trim.length());
      //System.out.println("  Kommentar: " + kommentar);
      if(kommentar.toLowerCase().startsWith("ignorespace")) {
        this.ignoriereLeerzeichen = true;
      }
      return start;
    }
    if(trim.startsWith("N:")) {
      this.neueNichtterminale(trim.substring(2).trim());
      return start;
    }
    if(trim.startsWith("T:")) {
      this.neueTerminale(trim.substring(2).trim());
      return start;
    }
    if(trim.startsWith("S:")) {
      start = trim.substring(2).trim();
      return start;
    }
    if(trim.contains("->")) {
      regeln.add(trim);
      return start;
    }
    //trim = trim.split("\\%")[0].trim();
    System.err.println("WARNUNG, Zeile nicht verarbeitbar: " + trim);
    return start;
  }
	 
  /** der String enthaelt mit Leerzeichen trennte Texte, die
   * in Nichtterminale umgewandelt werden.
   * @param s in Nichtterminale umzuwandelnder Text
   */
	 public void neueNichtterminale(String s) {
	   //System.out.println(" neu nt: " + s);
	   String[] alle = s.split("\\s+");
	   for(String st:alle) {
	     this.addNichtterminale(Nichtterminal.nichtterminal(st.trim()));
	   }
	 }
	 
	  /** der String enthaelt mit Leerzeichen trennte Texte, die
	   * in Terminale umgewandelt werden.
	   * @param s in Terminale umzuwandelnder Text
	   */
   public void neueTerminale(String s) {
     //System.out.println("Terminale: " + s);
     String[] alle = s.split("\\s+");
     for(String st:alle) {
       st = st.trim();
       if(st.equals("/space")) {
         st = " ";
       }
       this.addTerminale(Terminal.terminal(st));
     }
   }
   
   /** Jedes Element der uebergenenen String-Sammlung wird als eine
    * kontextfreie Regel in Textsyntax aufgefasst und zur Grammatik this
    * ergaenzt.
    * @param s Sammlung von Strings, die als Regeln interpretiert werden
    */
   public void neueRegeln(List<String> s) {
     for(String r:s) {
       String[] lr = r.split("->");
       if (lr.length != 2){
         System.err.println("WARNUNG, Regel nicht lesbar: " + r);
         continue;
       }
       String[] lrr = lr[0].split("::");
       String name = null;
       Nichtterminal links = null;
       if(lrr.length == 1) {
         links = Nichtterminal.nichtterminal(lr[0].trim());
       }
       if(lrr.length == 2) {
         //System.out.println(lrr[1] + " - " + lrr[0]);
         links = Nichtterminal.nichtterminal(lrr[1].trim());
         name = lrr[0].trim();
       }
       if(lrr.length == 0 || lrr.length > 2) {
         throw new IllegalStateException("Linke Seite darf nicht leer sein"
             + " und nur einmal :: enthalten");
       }
       //Wort w = stringAlsWort(lr[1].trim());
       String[] rechteSeiten = lr[1].trim().split("\\|");
       int pos = 0;
       for(String re:rechteSeiten) {
         Wort w = this.stringAlsWort(re.trim());
         if (w == null) {
           System.err.println("WARNUNG, Regel nicht lesbar: " + r);
           continue;
         }
         if (name != null && rechteSeiten.length > 1) {
           this.addRegeln(new KontextfreieRegel(links, w, name + "_" + (pos++)));
         } else {
           this.addRegeln(new KontextfreieRegel(links, w, name));
         }
       }
     }
   }
   

   /** Berechnet alle Nichtterminale, die letztendlich in Worte aus
    *  Terminalzeichen abgeleitet werden koennen.
    *  @return Nach Terminalzeichen ableitbare Nichtterminalzeichen
    */
   public List<Nichtterminal> nachTerminalenAbleitbar(){
     List<KontextfreieRegel> erg = new ArrayList<>();
     List<Nichtterminal> terminierbar = new ArrayList<>();
     int alt = -1;
     while(erg.size() != alt) {
       alt = erg.size();
       for(KontextfreieRegel reg: this.regeln) {
         if(!erg.contains(reg)) {
           boolean ok = true;
           for(Zeichen z: reg.getRechts()) {
             if(!(z instanceof Terminal) && !terminierbar.contains(z)) {
               ok = false;
               break;
             }
           }
           if(ok) {
             erg.add(reg);
             if(!terminierbar.contains(reg.getLinks())) {
               terminierbar.add(reg.getLinks());
             }
           }
         }
       }
     }
     return terminierbar;
   }
   
   /** Berechnet ein Liste aller Worte ueber die bekannten 
    * Terminale bis zur Laenge n einschliesslich, die mit dieser
    * Grammatik abgeleitet werden koennen.
    * @param n maximale Wortlaenge
    * @return Liste aller ableitbaren Worte bis zur Laenge n einschliesslich
    */
   public List<Wort> alleAbleitbarenWortBisZurLaenge(int n) {
     List<Wort> ergebnis = new ArrayList<Wort>();
     for(Wort w: Terminal.alleWorteBisLaenge(n)) {
       if (this.ableitbar(w).getAbgeleitet().equals(w)) {
         ergebnis.add(w);
       }
     }
     return ergebnis;
   }
   
   /** Berechnet alle Regeln, die letztendlich zu Woerterb abgeleitet 
    * werden koennen, die nur aus Terminalen und den uebergebenen
    * Nichtterminalen bestehen.
    * @param nterm Menge von erlaubten Nichtterminalen
    * @return
    */
   public List<KontextfreieRegel> ausStartNutzbar(List<Nichtterminal> nterm){
     List<KontextfreieRegel> erg = new ArrayList<>();
     Set<Nichtterminal> nutzbar = new HashSet<>();
     if(nterm.contains(this.start)) {
       nutzbar.add(this.start);
     } else {
       return erg;
     }
     int alt = -1;
     while(erg.size() != alt) {
       alt = erg.size();
       for(KontextfreieRegel reg: this.regeln) {
         if(!erg.contains(reg) && nutzbar.contains(reg.getLinks())) {
           boolean ok = true;
           for(Zeichen z: reg.getRechts()) {
             if(!(z instanceof Terminal) && !nterm.contains(z)) {
               ok = false;
               break;
             }
           }
           if(ok) {
             erg.add(reg);
             for(Zeichen z: reg.getRechts()) {
               if((z instanceof Nichtterminal) && !nutzbar.contains(z)) {
                 nutzbar.add((Nichtterminal) z);
               }
             }
           }
         }
       }
     }
     return erg;
   }
   
   /** Es werden alle Regeln geloescht, die bei dem aktuell gegebenen
    * Startsymbol niemals so genutzt werden koennen, dass letztendlich
    * ein Wort aus Terminalzeichen erreicht wird.
    */
   public void loescheUnnoetigeRegeln() {
     //System.out.println("loescheRegeln1");
     List<Nichtterminal> tmp2 = this.nachTerminalenAbleitbar();
     //System.out.println("loescheRegeln2");
     List<KontextfreieRegel> tmp1 = this.ausStartNutzbar(tmp2);
     //System.out.println("loescheRegeln3");
     this.regeln = tmp1;
   }
   
   
   /**
    * Verwandelt uebergebenen String in ein Wort, dabei werden von links
    * nach rechts Terminal- und Nichtterminalzeichen gesucht. /eps wird
    * zum leeren Wort, /space zu einem terminalen Leerzeichen, insofern
    * nicht this.ignoriereLeerzeichen gesetzt ist. Ist das der Fall werden
    * alle Leerzeichen geloescht; sollte die Loeschung nicht gewuenscht 
    * sein, ist stattdessen der Konstruktor new Wort(String) zu nutzen. 
    * Sollte die einfache Zuordnung nicht moeglich sein, wird eine
    * Warnung ausgegeben.
    * @param s umzuwandelnder String
    * @return resultierendes Wort
    */
   public Wort stringAlsWort(String s) {
     //return new Wort(s);
     Wort erg = new Wort();
     if(this.ignoriereLeerzeichen) {// && !Terminal.istTerminal(" ")) {
       s = s.replaceAll("\\s+", "");
     }
     if("/eps".equals(s) || "".equals(s)) {
       return erg;
     }
     String gefunden = "";
     for (int i = 0; i < s.length(); i++) {
       gefunden += s.charAt(i);
       if(gefunden.equals("/space")) {
         gefunden = " ";
       }
       if(Zeichen.istZeichen(gefunden)) {
         Zeichen t = Zeichen.zeichen(gefunden);
         gefunden ="";
         erg.anfuegen(t);
       }
     }
     if (!"".equals(gefunden)) {
       System.err.println("WARNUNG, " + s + " nicht vollstaendig "
           + "umgewandelt, fehlt " + gefunden);
     }   
     return erg;
   }
	
   
   /** Es wird die zur uebergebenen Id gehoerende Regel zurueckgegeben.
    * Sollte die Regel nicht existieren wird eine IllegalStateException
    * geworfen.
    * @param id Id der zu suchenden Regel
    * @return zur id passende Regel
    */
   public KontextfreieRegel regelZuId(String id) {
     for (KontextfreieRegel r: this.regeln) {
       if (id.equals(r.getId())){
         return r;
       }
     }
     throw new IllegalStateException("keine Regeln zum Namen " + id 
         + " gefunden");
   }
   
  /**
    * Ermoeglicht es eine Grammatik interaktiv auszufuehren, dabei werden
    * alle Regeln zur Auswahl und das aktuelle Wort angezeigt. Die 
    * Ausfuehrung endet, wenn das erreichte Wort nur noch aus 
    * Terminalzeichen bresteht. Die Ausfuehrung kann auch abgebrochen
    * werden. Als Ergebnis wird eine Liste der genutzen Regelausfuehrungen
    * (welche Regel, wo) zurueckgegeben.
   * @return Liste der anwegwandten Regelausfuehrungen
   */
	public List<Regelnutzung> ausfuehren() {
	  List<Regelnutzung> erg = new ArrayList<>();
		Wort w = new Wort(this.start);
		int eingabe = 0;
		while(!w.terminal()) {
			System.out.print(this 
			    + " (0) abbrechen\n"  
			    + "aktuelles Wort: " + w + " (" + w.laenge() + ")"
			    + "\n\nWelche Regel wollen Sie anwenden: ");
			eingabe = Eingabe.leseInt() - 1;
			if(eingabe == -1) {
			  break;
			}
			if (eingabe < 0 || eingabe >= this.regeln.size()) {
				System.out.println("Beinahe witzig.");
			}else {
				KontextfreieRegel reg = this.regeln.get(eingabe);
				List<Integer> pos = w.positionenVon(reg.getLinks());
				if (pos.isEmpty()) {
					System.out.println(" Regel nicht anwendbar.");
				} else {
					if(pos.size() == 1) {
						w.ersetzenVonMitNummer(reg.getLinks(), reg.getRechts(), 1);
						erg.add(new Regelnutzung(reg, pos.get(0)));
					} else {
						System.out.print("Weches Vorkommen (erstes ist 1): ");
						int nr = Eingabe.leseInt();
						w.ersetzenVonMitNummer(reg.getLinks(), reg.getRechts(), nr);
						erg.add(new Regelnutzung(reg, pos.get(nr-1)));
					}
				}
				//System.out.println("aktueller Text: " + w);
			}	
		}
		System.out.println("Ergebnis: " + w);
		return erg;
	}
	
	
	/** 
	 * Existierende Terminal-Objekte werden zu den Terminalzeichen dieser
	 * Grammatik hinzugefuegt.
	 * @param terminals hinzuzufuegende Terminalzeichen
	 * @return this
	 */
	public KontextfreieGrammatik addTerminale(Terminal... terminals) {
		terminale.addAll(Arrays.asList(terminals));
		return this;
	}
	
	/** 
   * Existierende Nichtterminal-Objekte werden zu den 
   * Nichtterminalzeichen dieser Grammatik hinzugefuegt.
   * @param nterminals hinzuzufuegende Nichtterminalzeichen
   * @return this
   */
	public KontextfreieGrammatik addNichtterminale(Nichtterminal... nterminals) {
		nichtterminale.addAll(Arrays.asList(nterminals));
		return this;
	}
	
	/**
	 * Uebergebene Regeln werden zu dieser Grammatik hinzugefuegt, dabei
	 * vorher auf syntaktiche Korrektheit geprueft. Es duerfen nur
	 * bereits bekannte Terminal- und Nichtterminalzeichen genutzt werden,
	 * beim Verstoss wird eine IllegalStateException geworfen.
	 * @param neu hinzuzufuegende Regeln
	 * @return this
	 */
	public KontextfreieGrammatik addRegeln(KontextfreieRegel... neu) {
	  List<KontextfreieRegel> dazu = new ArrayList<>();
	  for(KontextfreieRegel reg: neu) {
	    //System.out.println("neu: " + reg);
	    if(!this.nichtterminale.contains(reg.getLinks())) {
	      throw new IllegalStateException("linke Seite von " + reg 
	          + " nicht in Nichterminalen");
	    }
	    for(Zeichen z: reg.getRechts()) {
	      if(z instanceof Terminal && !this.terminale.contains(z)) {
	        throw new IllegalStateException(z + " aus rechter Seite von " + reg 
	            + " nicht in Terminalen");
	      }
	      if(z instanceof Nichtterminal && !this.nichtterminale.contains(z)) {
          throw new IllegalStateException(z + " aus rechter Seite von " + reg 
              + " nicht in Nichtterminalen");
        }
	    }
	    if(!this.regeln.contains(reg)) {
	      dazu.add(reg);
	    }
	  }
    for(KontextfreieRegel reg: this.regeln) {
      for(KontextfreieRegel reg2: dazu) {
        if(reg.getId() != null && reg2.getId() != null && reg.getId().equals(reg2.getId())) {
          throw new IllegalStateException("Zwei Regeln mit gleichem Namen "
              + reg.getId()+": " + reg + ", " + reg2);
        }
      }
    }
		regeln.addAll(dazu);
		return this;
	}
	
	/**
	 * Ermoeglicht das Setzen des Startymbols, das aus den bekannten
	 * Nichtterminalzeichen kommen muss; sonst wird eine
	 * IllegalStateException geworfen.
	 * @param nt neues Startsymbol
	 * @return this
	 */
	public KontextfreieGrammatik setStart(Nichtterminal nt) {
		if (! nichtterminale.contains(nt)) {
			throw new IllegalStateException(nt + " nicht in Nichterminalen " + this.nichtterminale + "\n" + this);
		}
		this.start = nt;
		return this;
	}

	@Override
	public String toString() {
		StringBuffer sb = new StringBuffer( "Terminale =" + terminale 
				+ "\nNichtterminale = " + nichtterminale 
		        + "\nStart = " + start + "\n");
		for(int i = 0; i < this.regeln.size(); i++) {
			sb.append(" (" + (i+1) + ") " + this.regeln.get(i) + "\n");
		}
		return sb.toString();
	}
	
	/**
	 * Es wird die Menge der Nichtterminalzeichen berechnet, die direkt
	 * oder ueber mehrere Schritte zum leeren Wort abgeleitet werden
	 * koennen.
	 * @return Menge von Nichtterminalen, die nach dem leeren Wort ableitbar sind
	 */
	public Set<Nichtterminal> nachLeeremWortAbleitbar(){
	  Set<Nichtterminal> erg = new HashSet<>();
	 
    for(KontextfreieRegel r: this.regeln) {
      if (r.getRechts().laenge() == 0) {
        erg.add(r.getLinks());
      }
    }
    if(erg.isEmpty()) {
      return erg;
    }
    
    boolean ende = false;
	  while(!ende) {
	    ende = true;
	    for(KontextfreieRegel r: this.regeln) {
	      if (r.getRechts().bestehtAus(erg) && !erg.contains(r.getLinks())) {
	        erg.add(r.getLinks());
	        ende = false;
	      }
	    }
	  }
	  return erg;
	}
	
	 /**
   * Es wird die Menge der Nichtterminalzeichen berechnet, die direkt
   * oder ueber mehrere Schritte zum leeren Wort abgeleitet werden
   * koennen.
   * @return Menge von Nichtterminalen, die nach dem leeren Wort ableitbar sind
   */
	 public Map<Nichtterminal, CYKKnoten> nachLeeremWortAbleitbarCYK(){
	    // evtl umbenennung da erg nicht mehr Ergebnis
	    Set<CYKKnoten> erg = new HashSet<>();
	    Map<Nichtterminal, CYKKnoten> zwischen = new HashMap<>();
	   
	    for(KontextfreieRegel r: this.regeln) {
	      if (r.getRechts().laenge() == 0) {
	        CYKKnoten tmp = new CYKKnoten(r.getLinks(), new Regelnutzung(r, 0));
	        erg.add(tmp);
	        zwischen.put(r.getLinks(), tmp);
	      }
	    }
	    if(erg.isEmpty()) {
	      return zwischen;
	    }
	    
	    boolean ende = false;
	    while(!ende) {
	      ende = true;
	      Set<Nichtterminal> leer = zwischen.keySet();
	      for(KontextfreieRegel r: this.regeln) {
	       
	        //System.out.println(leer);
	        if (r.getRechts().bestehtAus(leer) && !leer.contains(r.getLinks())) {
	          //erg.add(r.getLinks());
	          var tmp = new CYKKnoten(r.getLinks(), new Regelnutzung(r, 0));
	          for(Zeichen z: r.getRechts()) {
	            tmp.hinzu(zwischen.get(z).getRegeln());
	          }
	          zwischen.put(r.getLinks(), tmp);
	          //System.out.println(zwischen);
	          ende = false;
	        }
	      }
	    }
	    zwischen.forEach((k,v) -> {
	      if (v.getRegeln().size() > 1) {
	        var reg = new KontextfreieRegel(k, new Wort());
	        ergaenzteRegeln.put(reg, v.getRegeln());
	        // System.out.println(" leer: " + reg + " : " + v.getRegeln());
	    }      
	    });
	    
	    return zwischen;
	  }
	
	/**
	 * Es werden alle Regeln entfernt, bei denen ein Nichtterminal
	 * auf der rechten Seite zum leeren Wort abgeleitet werden kann
	 * und neue Regeln ergaenzt, die die rechte Seite ohne dieses
	 * Nichtterminal enthalten. Alle Regeln mit dem leeren Wort auf
	 * der rechten Seite werden geloescht.
	 * Falls das leere Wort in der Sprache liegt, ist das Ergebnis
	 * true sonst false.
	 * @return gehoert leeres Wort zu Sprache?
	 */
	public boolean ohneLeeresWortRegeln() {
	  Set<Nichtterminal> nachLeer = this.nachLeeremWortAbleitbar();
	  //System.out.println("nachleer0: " + nachLeer);
	  int tmp = 0;
	  while (tmp != this.regeln.size()) {
	    tmp = this.regeln.size();
	    //System.out.println("tmp: " + tmp);
	    List<KontextfreieRegel> neu = new ArrayList<>();
	    for(KontextfreieRegel r: this.regeln) {
	      for(Nichtterminal z: nachLeer) {
	        //System.out.println("z: " + z);
	        List<Integer> positionen = r.getRechts().positionenVon(z);
	        for(int i = positionen.size() - 1; i >= 0; i--) {
	          KontextfreieRegel reg = new KontextfreieRegel(r.getLinks()
	              , r.getRechts().clone().ersetzeAnPositionMit(
	                  positionen.get(i), new Wort()));
	          if (!this.regeln.contains(reg) && ! neu.contains(reg)) {
	            neu.add(reg);
	          }
	        }	        
	      }
	    }
	    this.regeln.addAll(neu);
	  }
	  for(int i = this.regeln.size() - 1; i >= 0; i--) {
	    if (this.regeln.get(i).getRechts().equals(new Wort())) {
	      this.regeln.remove(i);
	    }
	  }
	  //System.out.println("nachleer1: " + nachLeer);
	  return nachLeer.contains(this.start);
	}
	
	public boolean ohneLeeresWortRegelnCYK() {
	  Map<Nichtterminal, CYKKnoten> ableitungen = this.nachLeeremWortAbleitbarCYK();
    Set<Nichtterminal> nachLeer = ableitungen.keySet();
    //System.out.println("nachleer0: " + nachLeer);
    int tmp = 0;
    while (tmp != this.regeln.size()) {
      tmp = this.regeln.size();
      //System.out.println("tmp: " + tmp);
      List<KontextfreieRegel> neu = new ArrayList<>();
      for(KontextfreieRegel r: this.regeln) {
        for(Nichtterminal z: nachLeer) {
          //System.out.println("z: " + z);
          List<Integer> positionen = r.getRechts().positionenVon(z);
          for(int i = positionen.size() - 1; i >= 0; i--) {
            KontextfreieRegel basis = r;
            KontextfreieRegel reg = new KontextfreieRegel(r.getLinks()
                , r.getRechts().clone().ersetzeAnPositionMit(
                    positionen.get(i), new Wort()));
            if (!this.regeln.contains(reg) && ! neu.contains(reg)) {
              neu.add(reg);
              
              List<Regelnutzung> lrn = new ArrayList<>();
              lrn.add(new Regelnutzung(basis, 0));
              for(Regelnutzung rn: ableitungen.get(z).getRegeln()) {
                lrn.add(new Regelnutzung(rn.getRegel(), positionen.get(i)));
              }
              ergaenzteRegeln.put(reg, lrn);
              //c
              basis = reg;
            }
          }         
        }
      }
      this.regeln.addAll(neu);
    }
    for(int i = this.regeln.size() - 1; i >= 0; i--) {
      if (this.regeln.get(i).getRechts().equals(new Wort())) {
        this.regeln.remove(i);
      }
    }
    //System.out.println("nachleer1: " + nachLeer);
    return nachLeer.contains(this.start);
  }
	
	/**
	 * Es werden alle Kettenregeln, mit denen ein Nichtterminalzeichen
	 * auf ein andreres Nichtterminalzeichen abgeleitet werden kann,
	 * entfernt und aequivalente Regeln ohne diese Kette ergaenzt
	 * @return this mit gleicher erzeugter Sprache ohne Kettenregeln
	 */
	public KontextfreieGrammatik entferneKettenregeln() { 
	  //System.out.println("entferne Ketten1: " + this);
	  List<Paar> kette = new ArrayList<>();
	  for(Nichtterminal n: Nichtterminal.alleNichtterminale()) {
	    kette.add(new Paar(n, n));
	  }
	  int tmp = 0;
	  while (tmp != kette.size()) {
	    tmp = kette.size();
	    List<Paar> neu = new ArrayList<>();
	    for(Paar p1:kette) {
	      for(KontextfreieRegel r: this.regeln) {
	        if (r.getLinks() == p1.rechts 
	            && r.getRechts().istNichtterminal()) {
	          Paar pa = new Paar(p1.links, r.getRechts().alsNichtterminal());
	          if(!kette.contains(pa) && !neu.contains(pa)) {
	            neu.add(pa);
	          }
	        }
	      }
	    }
	    kette.addAll(neu);
	  }
	  //System.out.println("kette: " + kette);
	  
	  List<KontextfreieRegel> erg = new ArrayList<>();
	  for(Paar pa: kette) {
	    for(KontextfreieRegel re: this.regeln) {
	      if (re.getLinks() == pa.rechts && !re.getRechts().istNichtterminal()) {
	        KontextfreieRegel neu = new KontextfreieRegel(pa.links, re.getRechts());
	        if(!erg.contains(neu)) {
	          erg.add(neu);
	        }
//	      } else {
//	        erg.add(re);
	      }
	    }
	  }
	  
	  this.regeln = erg;
	  
	  // System.out.println("entferne Ketten2: " + this);
	  return this;
	}
	
	 /**
   * Es werden alle Kettenregeln, mit denen ein Nichtterminalzeichen
   * auf ein andreres Nichtterminalzeichen abgeleitet werden kann,
   * entfernt und aequivalente Regeln ohne diese Kette ergaenzt.
   * Intern werden die Regeln in CYKKonoten vermerkt, die zum Loeschen der
   * Kettenregeln genutzt werden.
   * @return this mit gleicher erzeugter Sprache ohne Kettenregeln
	 */
	@SuppressWarnings("unchecked")
  public KontextfreieGrammatik entferneKettenregelnCYK() { 
    //System.out.println("entferne Ketten1: " + this);
    List<Paar> kette = new ArrayList<>();
    for(Nichtterminal n: Nichtterminal.alleNichtterminale()) {
      kette.add(new Paar(n, n));
    }
    int tmp = 0;
    while (tmp != kette.size()) {
      tmp = kette.size();
      List<Paar> neu = new ArrayList<>();
      for(Paar p1:kette) {
        for(KontextfreieRegel r: this.regeln) {
          if (r.getLinks() == p1.rechts 
              && r.getRechts().istNichtterminal()) {
            Paar pa = new Paar(p1.links, r.getRechts().alsNichtterminal());
            if(!kette.contains(pa) && !neu.contains(pa)) {
              neu.add(pa);
              pa.regeln = (ArrayList<Regelnutzung>)p1.regeln.clone();
              pa.hinzu(new Regelnutzung(r, 0));
            }
          }
        }
      }
      kette.addAll(neu);
    }
    //System.out.println("kette: " + kette);
    
    List<KontextfreieRegel> erg = new ArrayList<>();
    for(Paar pa: kette) {
      for(KontextfreieRegel re: this.regeln) {
        if (re.getLinks() == pa.rechts && !re.getRechts().istNichtterminal()) {
          KontextfreieRegel neu = new KontextfreieRegel(pa.links, re.getRechts());
          if(neu.equals(re)) {
            neu = re;  // keine unnoetigen 1-zu-1 Kopien
          }
          if(!erg.contains(neu)) {
            erg.add(neu);
            List<Regelnutzung> abl = (ArrayList<Regelnutzung>)pa.regeln.clone();
            abl.add(new Regelnutzung(re, 0));
            if(neu != re) {
              this.ergaenzteRegeln.put(neu, abl);
            }
          }
//        } else {
//          erg.add(re);
        }
      }
    }
    
    this.regeln = erg;
    
    // System.out.println("entferne Ketten2: " + this);
    return this;
  }
	
	/**
	 * Es wird eine sprachaeguivalente Grammatik konstruiert, die die 
	 * gegebene ersetzt, deren Regeln in Chomsky-Normalform sind
	 * @return this, umgeformt in Chomsky-Normalform
	 */
	public KontextfreieGrammatik inChomskyNormalform() {
	  boolean leeresWort = this.ohneLeeresWortRegeln();
	  //System.out.println("ohne leeres-Wort-Regeln:\n" + this + "\n");
	  this.entferneKettenregeln();
	  //System.out.println("ohne Kette:\n" + this + "\n");
	  
	  Set<Terminal> genutzt = new HashSet<>();
	  Map<Terminal, Nichtterminal> ersetzer = new HashMap<>();
	  for(Terminal t: Terminal.alleTerminale()) {
	    ersetzer.put(t, Nichtterminal.neu());
	    this.addNichtterminale(ersetzer.get(t));
	  }
	  for(KontextfreieRegel reg: this.regeln) {
	    Wort neu = new Wort();
	    if(reg.getRechts().laenge() > 1) {
  	    for(Zeichen z: reg.getRechts()) {
  	      if(z instanceof Nichtterminal) {
  	        neu.anfuegen(z);
  	      } else {
  	        neu.anfuegen(ersetzer.get((Terminal)z));
  	        genutzt.add((Terminal)z);
  	      }
  	    }
  	    reg.setRechts(neu);
	    }
	  }
	  
	  for(Terminal t: genutzt) {
	    this.addRegeln(new KontextfreieRegel(ersetzer.get(t), new Wort(t)));
	  }
	  
	  //System.out.println("Terminale ersetzt:\n" + this);
	  
	  List<KontextfreieRegel> neu = new ArrayList<>();
	  for(KontextfreieRegel reg: this.regeln) {
	    if (reg.getRechts().laenge() < 3) {
	      neu.add(reg);
	    } else {
	      Nichtterminal hilf = Nichtterminal.neu();
	      this.addNichtterminale(hilf);
	      Wort rechts = reg.getRechts();
	      neu.add(new KontextfreieRegel(reg.getLinks(), new Wort(rechts.at(0), hilf)));
	      for(int i = 1 ; i < rechts.laenge() - 2; i++) {
	        Nichtterminal hilf2 = Nichtterminal.neu();
	        this.addNichtterminale(hilf2);
	        neu.add(new KontextfreieRegel(hilf, new Wort(rechts.at(i), hilf2)));
	        hilf = hilf2;
	      }
	      neu.add(new KontextfreieRegel(hilf
	          , new Wort(rechts.at(rechts.laenge() - 2)
	              , rechts.at(rechts.laenge() - 1))));
	    }
	  }
	  this.regeln = neu;
	  
	  if(leeresWort) {
//	    Nichtterminal nt = Nichtterminal.neu();
//	    this.addNichttterminale(nt);
//	    this.addRegeln(new KontextfreieRegel(nt, new Wort()));
//	    this.addRegeln(new KontextfreieRegel(nt, new Wort(this.start)));
//	    this.start = nt;
	    this.addRegeln(new KontextfreieRegel(this.start, new Wort()));
	  }
	  return this;
	}
	
	 /**
   * Es wird eine sprachaeguivalente Grammatik konstruiert, die die 
   * gegebene ersetzt, deren Regeln in Chomsky-Normalform sind. Intern
   * werden Umformungen in CYK-Knoten vermerkt
   * @return this, umgeformt in Chomsky-Normalform
   */
	 public KontextfreieGrammatik inChomskyNormalformCYK() {
	    boolean leeresWort = this.ohneLeeresWortRegelnCYK();
	    // System.out.println("\nohne leeres-Wort-Regeln:\n" + this + "\n" + this.ergaenzteRegeln +"\n");
	    this.entferneKettenregelnCYK();
	    //System.out.println("ohne Kette:\n" + this + "\n");
	   
	    Set<Terminal> genutzt = new HashSet<>();
	    Map<Terminal, Nichtterminal> ersetzer = new HashMap<>();
	    for(Terminal t: Terminal.alleTerminale()) {
	      ersetzer.put(t, Nichtterminal.neu());
	      this.addNichtterminale(ersetzer.get(t));
	    }
	    
	    List<KontextfreieRegel> neueRegeln = new ArrayList<>();
	    
	    for(KontextfreieRegel reg: this.regeln) {
	      Wort neu = new Wort();
	      //if(reg.getRechts().laenge() > 1 && !(reg.getRechts().laenge() == 2 && reg.getRechts().ausNichtTerminalen())) {
	      if(reg.getRechts().laenge() > 1 && !reg.getRechts().ausNichtTerminalen()) {
	        for(Zeichen z: reg.getRechts()) {
	          if(z instanceof Nichtterminal) {
	            neu.anfuegen(z);
	          } else {
	            neu.anfuegen(ersetzer.get((Terminal)z));
	            genutzt.add((Terminal)z);
	          }
	        }
	        KontextfreieRegel r = new KontextfreieRegel(reg.getLinks(), neu);
	        //if(!reg.equals(r)) {
  	        //System.out.println(" ergaenzt: " + r + " : " + reg);
  	        ergaenzteRegeln.put(r, List.of(new Regelnutzung(reg, 0)));
  	        //System.out.println(" ergaenzt: " + r + " : " + ergaenzteRegeln.get(r));
  	        neueRegeln.add(r);
	        //} else {
	        //  neueRegeln.add(reg);
	        //}
	      } else {
	        neueRegeln.add(reg);
	      }
	    }
	    this.regeln = neueRegeln;
	    
	    
	    // da nur Terminale ersetzt werden, wird dies leer zu ergaenzteRegeln hinzugefuegt 
	    for(Terminal t: genutzt) {
	      var r = new KontextfreieRegel(ersetzer.get(t), new Wort(t));
	      this.addRegeln(r);
	      ergaenzteRegeln.put(r, List.of());
	    }
	    
	    //System.out.println("Terminale ersetzt:\n" + this);
	    
	    List<KontextfreieRegel> neu = new ArrayList<>();
	    for(KontextfreieRegel reg: this.regeln) {
	      if (reg.getRechts().laenge() < 3) {
	        neu.add(reg);
	      } else {
	        Nichtterminal hilf = Nichtterminal.neu();
	        this.addNichtterminale(hilf);
	        Wort rechts = reg.getRechts();
	        int pos = -1;
	        var r = new KontextfreieRegel(reg.getLinks(), new Wort(rechts.at(0), hilf));
	        neu.add(r);
	        this.ergaenzteRegeln.put(r,List.of(new Regelnutzung(reg, pos)));
	        
	        for(int i = 1 ; i < rechts.laenge() - 2; i++) {
	          Nichtterminal hilf2 = Nichtterminal.neu();
	          this.addNichtterminale(hilf2);
	          var r2 = new KontextfreieRegel(hilf, new Wort(rechts.at(i), hilf2));
	          this.ergaenzteRegeln.put(r2, List.of(new Regelnutzung(reg, pos - i)));
	          neu.add(r2);
	          hilf = hilf2;
	        }
	        var r3 = new KontextfreieRegel(hilf
              , new Wort(rechts.at(rechts.laenge() - 2)
                  , rechts.at(rechts.laenge() - 1)));
	        this.ergaenzteRegeln.put(r3, List.of(new Regelnutzung(reg, Integer.MIN_VALUE)));
	        neu.add(r3);
	      }
	    }
	    this.regeln = neu;
	    
	    if(leeresWort) {
//	      Nichtterminal nt = Nichtterminal.neu();
//	      this.addNichttterminale(nt);
//	      this.addRegeln(new KontextfreieRegel(nt, new Wort()));
//	      this.addRegeln(new KontextfreieRegel(nt, new Wort(this.start)));
//	      this.start = nt;
//	      System.out.println("\n*****\n ergaenze Regeln\n");
//	      this.ergaenzteRegeln.forEach((r,a) -> System.out.println(r +" : " + a));
//	      System.out.println("\n****\n");
	      KontextfreieRegel nreg = new KontextfreieRegel(this.start, new Wort());
	      if(!this.regeln.contains(nreg)) {
	          this.addRegeln(new KontextfreieRegel(this.start, new Wort()));
	          //System.out.println("in ergaenzt? " + this.ergaenzteRegeln.keySet().contains(nreg));
	      } else {
	        //System.out.println("contained");
	      }
	      //System.out.println("\n****\n");
	    }
	    
	    //System.out.println("\nEnde CNF:\n" + this + "\n" + this.ergaenzteRegeln +"\n");
	    return this;
	  }
	
	/**
	 * Prueft ob die gegebene Grammatik in Chomsky-Normalform ist.
	 * @return this in Chomsky-Normalform?
	 */
	public boolean istInChomskyNormalform() {
	  for (KontextfreieRegel reg: this.regeln) {
	    Wort rechts = reg.getRechts();
	    if(rechts.laenge() == 0 && reg.getLinks() != this.start) {
	      //System.out.println("Falsche epsilon-Regel: " + reg);
	      //System.out.println(this);
	      return false;
	    }
	    if(rechts.laenge() == 1 && !(rechts.at(0) instanceof Terminal)) {
	      //System.out.println("Bei einem Zeichen nur Terminal erlaubt: " + reg);
	      return false;
	    }
	    if(rechts.laenge() == 2 && (!(rechts.at(0) instanceof Nichtterminal)
	        || !(rechts.at(1) instanceof Nichtterminal))) {
	      // System.out.println("Bei zwei Zeichen nur zwei Nichtterminale erlaubt: " + reg);
        return false;
      }
	    if(rechts.laenge() > 2) {
	      //System.out.println("Keine Regeln mit mehr als zwei Zeichen: " + reg);
	      return false;
	    }
	  }
	  return true;
	}
	
	/**
	 * Liest ein Wort-Objekt aus einer gebenenen Datei datei, die einen
	 * einfachen Text enthaelt. 
	 * Bei Problemen wird ein eine Warnung ausgegeben.
	 * @param datei (Pfad und) Dateinamen, aus der das Wort gelesen wird
	 * @return resultierendes Wort
	 */
	public Wort leseWortAusDatei(String datei) {
	   File file = new File(datei);
	    if (file.exists() && file.canRead()) {
	      Path path = Paths.get(datei);
	      try (BufferedReader reader = Files.newBufferedReader(path, Charset.forName("UTF-8"))) {

	        String currentLine = null;
	        String text = "";
	        while ((currentLine = reader.readLine()) != null) {
	          if(currentLine.endsWith("\n") || currentLine.endsWith("\r")) {
	            currentLine = currentLine.substring(0, currentLine.length()-1);
	          }
	          if(currentLine.endsWith("\n") || currentLine.endsWith("\r")) {
	            currentLine = currentLine.substring(0, currentLine.length()-1);
	          }
	          text += currentLine;   
	        }
	        //System.out.println(this.ignoriereLeerzeichen + " pruefe " + text);
	        if(this.ignoriereLeerzeichen) {
	          text = text.replaceAll("\\s+", "");
	        }
	        //System.out.println("pruefe " + text);
	        return this.stringAlsWort(text);
	      } catch(Exception e) {
	        System.err.println("WARNUNG: Problem beim Lesen des Strings "
	            + "aus der Datei " 
	            + datei + ": " + e);
	      }
	    } else {
	      System.err.println("WARNUNG: Problem beim Zugriff auf " + datei);
	    }
	    return null;
	}
	
	/**
	 * Ist der Text aus der Datei datei aus dieser Grammatik ableitbar.
	 * @param datei (Pfad und) Dateiname mit zu ueberpruefenden Text
	 * @return ist Text aus Datei fatei ableitbar?
	 */
	public boolean ableitbarAusDatei(String datei) {
	  Wort w = this.leseWortAusDatei(datei);
	  if (w == null) {
	    return false;
	  }
	  return this.ableitbar(w).isErfolgreich();
	}
	
	/** fuer das uebergebene Wort wird eine zugehoerige Ableitung berechnet,
	 * sollte dies nicht moeglich sein, wird eine Ableitung fuer den 
	 * Wortanfang ausgegeben. Im Ergebnis ist festgehalten, ob die 
	 * Suche nach einer Ableitung erfolgreich war.
	 * @param w Wort, fuer das eine Ableitung gesucht wird
	 * @return berechnete Ableitung, zumindest fuer den ableitbaren Anfang
	 */
	public Ableitung berechneAbleitung(Wort w){
	  Ableitung regeln = this.ableitbar(w);
	  Ableitung tmp = null;
	  if(regeln == null) {
	    return null;
	  }
	  while(!regeln.equals(tmp)) {
	    tmp = regeln;
	    
//      for (Regelnutzung re : regeln) {
//        System.out.println(re);
//      }
      //System.out.println("------------------------------------------");
	      
	    regeln = this.urspruenglicheRegeln(regeln);

	  }
	  //System.out.println("regeln:\n"+ regeln + "\ntmp:\n" + tmp + "\n");
	  
	  // etwas schmutziger Trick, um wieder Linksableitung zu erhalten
	  return new Ableitung(regeln.getSequenz(), regeln.isErfolgreich(), regeln.getAbgeleitet());
	}
	
	private Ableitung urspruenglicheRegeln(Ableitung regeln){
	  if(regeln == null || regeln.getSequenz() == null) {
	    return new Ableitung(List.of(), false, new Wort());
	  }
	  Ableitung erg = new Ableitung(new ArrayList<>(), regeln.isErfolgreich(), regeln.getAbgeleitet());
	  List<KontextfreieRegel> ziele = new ArrayList<>();
	  for(int i = 0; i < regeln.size(); i++) {
	    var r = regeln.get(i);
	    List<Regelnutzung> ersatz = this.ergaenzteRegeln.get(r.getRegel());
	    if(ersatz == null) {
	      erg.add(r);
	      continue;
	    }
	    if(ersatz.size() == 0) {
	      continue;
	    }
	    Regelnutzung erste = ersatz.get(0);
	    if(erste.getPosition() >= 0) {
	      for(Regelnutzung neu:ersatz) {
	        erg.add(new Regelnutzung(neu.getRegel(), neu.getPosition() + r.getPosition()));
	      }
	    } else { // aus Kettenregelumformung
	      // Version 0, laeuft bei einigen Beispielen
//	      var ziel = erste.getRegel();
//	      while(
//	        i+1 < regeln.size()
//	        &&  this.ergaenzteRegeln.get(regeln.get(i+1).getRegel()) != null 
//	        && (this.ergaenzteRegeln.get(regeln.get(i+1).getRegel()).size() == 0
//	            || this.ergaenzteRegeln.get(regeln.get(i+1).getRegel()).get(0).getRegel().equals(ziel))
//	        && (this.ergaenzteRegeln.get(regeln.get(i+1).getRegel()).size() == 0
//	            || this.ergaenzteRegeln.get(regeln.get(i+1).getRegel()).get(0).getPosition() != -1	           )
//	        ){
//	        i++;
//	      }
//	      erg.add(new Regelnutzung(ziel, r.getPosition()));	   
	      
	      //var ziel = erste.getRegel();
        ziele.add(erste.getRegel());
        if (erste.getPosition() == -1) {
          erg.add(new Regelnutzung(erste.getRegel(), r.getPosition())); 
        }
//	      while(
//          i+1 < regeln.size()
//          &&  this.ergaenzteRegeln.get(regeln.get(i+1).getRegel()) != null 
//          && (this.ergaenzteRegeln.get(regeln.get(i+1).getRegel()).size() == 0
//              || ziele.contains(this.ergaenzteRegeln.get(regeln.get(i+1).getRegel()).get(0).getRegel()))
//          && (this.ergaenzteRegeln.get(regeln.get(i+1).getRegel()).size() == 0
//              || this.ergaenzteRegeln.get(regeln.get(i+1).getRegel()).get(0).getPosition() != Integer.MIN_VALUE             )
//          ){
//          i++;
//        }
//	      System.out.println("  um: " + r);
//        erg.add(new Regelnutzung(ziele.remove(0), r.getPosition()));       
	    }
	    
	  }
	  return erg;
	}
	
	
	 /**
   * Ist das uebergebene Wort mit dieser Grammatik ableitbar? Das Ergebnis
   * ist ein Ableitungsobjekt, dem auch der Erfolg der Ableitung entnommen
   * werden kann.
   * @param w zu ueberpruefendes Wort
   * @return Ableitung des Wortes oder eines Wortanfangs, wenn nicht
   *         vollstaendig ableitbar. Erfolg steht in Variable der
   *         Ableitung
   */
  @SuppressWarnings("unchecked")
  public Ableitung ableitbar(Wort w) {
    if(!this.istInChomskyNormalform()) {
      this.inChomskyNormalformCYK();
    }
    int size = w.laenge();
    
    if (size == 0) {
      KontextfreieRegel leer = new KontextfreieRegel(this.start, new Wort());
      if (this.regeln.contains(leer)){
        //System.out.println("---leeres Wort abgeleitet---");
        if (this.ergaenzteRegeln.get(leer) == null) {
          return new Ableitung(List.of(
            new Regelnutzung(new KontextfreieRegel(this.start, new Wort()), 0)),true,w);
        } else {
          return new Ableitung(this.ergaenzteRegeln.get(leer), true, w);
        }
      } else {
        return new Ableitung(List.of(), false, new Wort());
      }
    }
    
    HashSet<CYKKnoten>[][] n = (HashSet<CYKKnoten>[][])new HashSet[size][size];
    for(int i = 0; i < size; i++) {
      Zeichen z = w.at(i);
      n[i][0] = new HashSet<>();
      for(KontextfreieRegel reg: this.regeln) {
        if(reg.getRechts().laenge() > 0 && reg.getRechts().at(0).equals(z)) {
          n[i][0].add(new CYKKnoten(reg.getLinks(),new Regelnutzung(reg, i)));
        }      
      }
    }
    
    //System.out.println("[" + size + "]");
    for(int j = 1; j < size; j++) {
      // System.out.print("*" + (j%10==0?"\n":""));
      for(int i = 0; i < size - j; i++) {        
        n[i][j] = new HashSet<>();
        for(int k = 0; k < j; k++) {
          for(KontextfreieRegel reg: this.regeln) {
//            System.out.println("i:" + i + " k:" + k + ":" + n[i][k] 
//                + " i+k+1:" + (i+k+1) + " j-k:" + (j-k-1) + ":" + n[i+k+1][j-k-1]);
            for(CYKKnoten b: n[i][k]) {
              for(CYKKnoten c: n[i+k+1][j-k-1]) {
//                System.out.println("  " + reg.getRechts() + " " + b + c);
                if(reg.getRechts().equals(new Wort(b.getN(),c.getN()))) {
                  //n[i][j].add(reg.getLinks());
                  CYKKnoten tmp = new CYKKnoten(reg.getLinks(), new Regelnutzung(reg, i));
                  tmp.getRegeln().addAll(b.getRegeln());
                  tmp.getRegeln().addAll(c.getRegeln());
                  n[i][j].add(tmp);
                }
              }
            }        
          }
        }
//        System.out.println("i:" + i + " j:" + j + " :" + n[i][j]);
      }
    }

    // Tabelle sichern fuer Ausgabe
    this.cykTabelle = (HashSet<Nichtterminal>[][])new HashSet[size][size];
    for(int j = 0; j < size; j++) {
      for(int i = 0; i < size; i++) {
        this.cykTabelle[j][i] = new HashSet<>();
        if(n[j][i] != null ) {
          for(CYKKnoten kn: n[j][i]) {
            this.cykTabelle[j][i].add(kn.getN()); 
          }
        }
      }
    }
    
    for(CYKKnoten k: n[0][size - 1]) {
      if (k.getN().equals(this.start)) {
        return new Ableitung(k.getRegeln(), true, w);
      }
    }
    for(int i= size - 2; i >= 0; i--) {
      for(CYKKnoten k: n[0][i]) {
        if (k != null && k.getN().equals(this.start)) {
          return new Ableitung(k.getRegeln(), false, w.vonBis(0, i + 1));
        }
      }
    }
    return new Ableitung(List.of(), false, new Wort());
    //return n[0][size-1].contains(this.start);
  }
  
  /** Insdofern vorhanden, wird zu dem zuletzt uebergebenen Wort,
   * fuer das geprueft wurde, ob aus mit dirser Grammatik ableitbar ist,
   * die zugehoerige CYK-Tabelle in der Konsole ausgegeben.
   */
  public void cykTabelleAusgeben() {
    if (cykTabelle == null) {
      System.out.println("aktuell keine Tabelle vorhanden");
      return;
    }
    int size = this.cykTabelle.length;
    int max = 0;
    for (int j = 0; j < size; j++) {
      for (int i = 0; i < size; i++) {
        if (this.cykTabelle[i][j] != null && (this.cykTabelle[i][j].toString().length() > max)){
          max = this.cykTabelle[i][j].toString().length();
        }
      }
    }
    for (int j = 0; j < size; j++) {
      for (int i = 0; i < size; i++) {
        String text = this.cykTabelle[i][j] != null 
                        ? this.cykTabelle[i][j].toString() : "-";
        while (text.length() <= max) {
          text += " ";
        }
        System.out.print(text);
      }
      System.out.println();
    }
  }
  
  /**
   * Ist uebergebener String als Wort mit dieser Grammatik ableitbar?
   * @param w zu ueberpruefender String
   * @return ist Wort ableitbar?
   */
  public boolean ableitbar(String w) {
    //return this.ableitbar(this.stringAlsWort(w)).isErfolgreich();
    return this.ableitbar(new Wort(w)).isErfolgreich();
  }
	
	/**
   * Ist das uebergebene Wort mit dieser Grammatik ableitbar?
   * @param w zu ueberpruefendes Wort
   * @return ist Wort ableitbar?
   */
	public boolean ableitbarRekursiv(Wort w) {
	  if(!this.istInChomskyNormalform()) {
	    this.inChomskyNormalform();
	  }
	  List<KontextfreieRegel> ableitung = this.ableitbarRekursiv(this.start, w
	      , new ArrayList<KontextfreieRegel>());
	  //System.out.println(ableitung);
	  return ableitung != null && ableitung.size() != 0;
	}
	
	private int step = 1;
	
	private List<KontextfreieRegel> ableitbarRekursiv(Nichtterminal n, Wort w
	    , List<KontextfreieRegel> liste ) {
	  if(step++%10000 == 0) {
	    System.out.println("." + this.regeln.size() + " " + n + " " + w);
	  }
	  if(w.laenge() == 0) {
      for (KontextfreieRegel reg: this.regeln) {
        if (reg.getLinks() == n && reg.getRechts().equals(w)) {
          liste.add(reg);
          //System.out.println(liste);
          return liste;
        }
      }
      return liste;
    }
	  
	  if(w.laenge() == 1) {
	    for (KontextfreieRegel reg: this.regeln) {
	      if (reg.getLinks() == n && reg.getRechts().equals(w)) {
	        liste.add(reg);
	        //System.out.println(liste);
	        return liste;
	      }
	    }
	    return null;
	  }
	    
    for (KontextfreieRegel reg: this.regeln) {
      if (reg.getLinks() == n && reg.getRechts().laenge() == 2) {
        for(int i = 1; i < w.laenge(); i++) {
          List<KontextfreieRegel> tmp = new ArrayList<>();
          if (this.ableitbarRekursiv((Nichtterminal)reg.getRechts().at(0)
              , w.vonBis(0, i), tmp) != null
              && this.ableitbarRekursiv((Nichtterminal)reg.getRechts().at(1)
                  , w.vonBis(i, w.laenge()), tmp) != null
            ) {
            // System.out.println(" " + reg + " " + w.vonBis(0, i) + " " + w.vonBis(i, w.laenge()));
            liste.addAll(tmp);
            liste.add(reg);
            //System.out.println(liste);
            return liste;
          }
        }
      }
    }
	  
	  return null;
	}
	
	/** Fuehrt die uebergeben Ableitung ausgehend vom Startsymbol aus
	 * und gibt die Berechneten Zwischenergebnisse und das Endergebnis
	 * in der Konsole aus.
	 * @param ab darzustellende und auszufuehrende Ableitung 
	 */
	public void ableitungAusfuehren(Ableitung ab) {
	  this.ableitungAusfuehren(this.start, ab);
	}
	
	private void ableitungAusfuehren(Nichtterminal nt, Ableitung ab) {
	  //System.out.println(nt);
	  Wort w = new Wort(nt);
	  for(Regelnutzung rn: ab) {
	    w = rn.getRegel().anwendenAnPositionAuf(rn.getPosition(), w);
	    //System.out.println(rn.getRegel() + " : " + w);
	  }
	}
	
	/** Ueberprueft ob die gegebene Grammatik rechtslinear ist.
	 * 
	 * @return ist rechtslinear?
	 */
	public boolean istRechtslinear() {
	  for(KontextfreieRegel reg: this.regeln) {
	    Wort tmp = reg.getRechts();
	    if(tmp.laenge() > 1 && !tmp.ausNichtTerminalen()) {
	      for(int i = 0; i < tmp.laenge() - 1; i++) { //Nichtterminal mitten drin
	        if(tmp.at(i).getClass() == Nichtterminal.class) {
	          return false;
	        }
	      }
	    }
	  }
	  return true;
	}
	
	/** Verwandelt eine rechtslineare Grammatik in einen endlichen
	 * Automaten. Sollte die Grammatik nicht rechtslinear sein, wird
	 * eine IllegalStatementException geworfen. 
	 * @return endlicher Automat, der Sprache akzeptiert, die die
	 *         Grammatik generiert
	 * @throws IllegalStateException        
	 */
	public EndlicherAutomat rechtslinearInAutomaten() {
	  if(! this.istRechtslinear()) {
	    throw new IllegalStateException("nur rechtslineare Grammatiken"
	        + " koennen in einen Automaten umgewandelt werden");
	  }
	  EndlicherAutomat ergebnis = new EndlicherAutomat();
	  for(Terminal t: this.terminale) {
	    ergebnis.addZeichen(t);
	  }
	  // neue Zustaende fuer Nichtterminale?
	  for(Nichtterminal nt: this.nichtterminale) {
	    ergebnis.addZustand(nt.getZeichen());
	  }
	  Zustand start = Zustand.zustand(this.start.getZeichen());
	  ergebnis.setStart(start);
	  for(KontextfreieRegel reg: this.regeln) {
	    Zustand von = Zustand.zustand(reg.getLinks().getZeichen());
      Wort tmp = reg.getRechts();
      //System.out.println(" starte Regel: " + reg);
      if (tmp.laenge() == 0) {
        ergebnis.addEndzustand(von);
      } else {
        Zustand aktuell = von;
        Zustand neu = null;
        for(int i = 0; i < tmp.laenge() - 1; i++) {
          neu = Zustand.neu();
          //System.out.println(" neu: " + aktuell + " " + tmp.at(i) + " " + neu);
          ergebnis.addUeberfuehrung(aktuell, (Terminal)tmp.at(i), neu);
          ergebnis.addZustand(neu);
          aktuell = neu;
        }
        Zeichen ende = tmp.at(tmp.laenge() - 1);
        //System.out.println(" z: " + ende + " " + ende.getClass());
        if(ende.getClass() == Terminal.class) {
          neu = Zustand.neu();
          //System.out.println(" neu2:" + aktuell + " " + ende + " " + neu);
          ergebnis.addUeberfuehrung(aktuell, (Terminal)ende, neu);
          ergebnis.addZustand(neu);
          ergebnis.addEndzustand(neu);
        } else {
          //System.out.println(" neu: " + aktuell + " eps " + ende);
          ergebnis.addEpsilonueberfuehrung(aktuell, Zustand.zustand(ende.getZeichen()));
        }
      }
      //System.out.println(" ende Regel ____");
	  }
	  return ergebnis;
	}
	
	/** Es wird zur als Wort interpretierten Eingabe eine Linksableitung 
   * auf der Konsole augegeben, wenn dies moeglich ist;
   * sonst wird eine Fehlermeldung als Text ausgegeben.
   * @param wort Wort zu dem Linksableitung gesucht wird
   */
  public void zeigeLinksableitung(String wort) {
    this.zeigeLinksableitung(new Wort(wort));
  }
	
	/** Es wird zum Wort eine Linksableitung auf der Konsole augegeben,
	 * wenn dies moeglich ist;
	 * sonst wird eine Fehlermeldung als Text ausgegeben.
	 * @param wort Wort zu dem Linksableitung gesucht wird
	 */
	public void zeigeLinksableitung(Wort wort) {
	  Ableitung ab = this.berechneAbleitung(wort);
	  if(!ab.isErfolgreich()) {
	    System.err.println(wort + " ist nicht ableitbar, nur: " + ab.getAbgeleitet());
	    return;
	  }
	  Ableitungsbaum abl = Ableitungsbaum.ableitungAlsBaum(ab);
    abl.linksdurchlauf();
	}
	
  /** Es wird zum als String uebergebenen Wort ein Ableitungsbaum 
   * ausgegeben, wenn dies moeglich ist;
   * sonst wird eine Fehlermeldung als Text ausgegeben.
   * @param wort String zu dem als Wort ein Ableitungsbaum gesucht wird
   */
  public void zeigeAbleitungsbaum(String wort) {
    this.zeigeAbleitungsbaum(new Wort(wort));
  }
	
	 /** Es wird zum Wort ein Ableitungsbaum ausgegeben,
   * wenn dies moeglich ist;
   * sonst wird eine Fehlermeldung als Text ausgegeben.
   * @param wort Wort zu dem Ableitungsbaum gesucht wird
   */
  public void zeigeAbleitungsbaum(Wort wort) {
    Ableitung ab = this.berechneAbleitung(wort);
    if(!ab.isErfolgreich()) {
      System.err.println(wort + " ist nicht ableitbar, nur: " + ab.getAbgeleitet());
      return;
    }
    Ableitungsbaum abl = Ableitungsbaum.ableitungAlsBaum(ab);
    abl.visualisieren();
  }

	@Override
	public KontextfreieGrammatik clone() {
	  KontextfreieGrammatik erg = new KontextfreieGrammatik();
	  erg.terminale = new HashSet<>(this.terminale);
	  erg.nichtterminale = new HashSet<>(this.nichtterminale);
	  erg.setStart(this.start);
	  for(KontextfreieRegel r: this.regeln) {
	    erg.addRegeln(r.clone());
	    //System.out.println("cloned: " + r);
	  }
	  //erg.regeln = tmp;
	  return erg;
	}

  public Set<Terminal> getTerminale() {
    return terminale;
  }

  public void setTerminale(Set<Terminal> terminale) {
    this.terminale = terminale;
  }

  public Set<Nichtterminal> getNichtterminale() {
    return nichtterminale;
  }

  public void setNichtterminale(Set<Nichtterminal> nichtterminale) {
    this.nichtterminale = nichtterminale;
  }

  public List<KontextfreieRegel> getRegeln() {
    return regeln;
  }

  public void setRegeln(List<KontextfreieRegel> regeln) {
    this.regeln = regeln;
  }

  public int getStep() {
    return step;
  }

  public void setStep(int step) {
    this.step = step;
  }

  public Nichtterminal getStart() {
    return start;
  }

  public Map<KontextfreieRegel, List<Regelnutzung>> getErgaenzteRegeln() {
    return ergaenzteRegeln;
  }

  public void setErgaenzteRegeln(
      Map<KontextfreieRegel, List<Regelnutzung>> ergaenzteRegeln) {
    this.ergaenzteRegeln = ergaenzteRegeln;
  }

  public boolean isIgnoriereLeerzeichen() {
    return ignoriereLeerzeichen;
  }

  public void setIgnoriereLeerzeichen(boolean ignoriereLeerzeichen) {
    this.ignoriereLeerzeichen = ignoriereLeerzeichen;
  }
  
  // gibt Alternativen, aber so klarer Zugriff
  class Paar{
    public Nichtterminal links;
    public Nichtterminal rechts;
    public ArrayList<Regelnutzung> regeln;
    
    public Paar(Nichtterminal links, Nichtterminal rechts) {
      super();
      this.links = links;
      this.rechts = rechts;
      this.regeln = new ArrayList<>();
    }
    
    public Paar hinzu(Regelnutzung r) {
      this.regeln.add(r);
      return this;
    }

    @Override
    public int hashCode() {
      return Objects.hash(links, rechts);
    }

    @Override // nicht regeln nutzen, da fuer Gleichheit nur Nichtterminale wichtig
    public boolean equals(Object obj) {
      if (this == obj)
        return true;
      if (obj == null)
        return false;
      if (getClass() != obj.getClass())
        return false;
      Paar other = (Paar) obj;
      return Objects.equals(links, other.links)
          && Objects.equals(rechts, other.rechts);
    }

    @Override
    public String toString() {
      return "Paar [links=" + links + ", rechts=" + rechts + ", regeln=" + regeln + "]";
    }
  }
  
}
