package regulaererAusdruck;

import java.io.Serializable;

import alphabet.Terminal;
import alphabet.Wort;
import ausfuehrung.Ableitung;
import endlicherAutomat.EndlicherAutomat;
import grammatik.KontextfreieGrammatik;
import semantik.Ableitungsbaum;

public abstract class RegulaererAusdruck implements Cloneable, Serializable{
  private static final long serialVersionUID = -3273703476382856371L;
  private static String grammatik = """
      %ignorespace
      N: Reg
      T: a b c d e f g h i j k l m n o p q r s t u v w x y z 0 1 2 3 4 5 6 7 8 9 ( ) * + {}
      S: Reg
      r1:: Reg -> (Reg)* | (Reg+Reg) | RegReg | (Reg) 
      r2:: Reg -> {} | a | b | c | d | e | f | g | h | i | j | k | l | m | n | o | p | q | r | s | t | u | v | w | x | y | z | 0 | 1 | 2 | 3 | 4 | 5 | 6 | 7 | 8 | 9
      """;

  private static KontextfreieGrammatik kfg;
  private static String[] fin = {"{}", "a", "b", "c", "d", "e", "f"
      , "g", "h", "i", "j", "k", "l", "m", "n", "o", "p", "q", "r"
      , "s", "t", "u", "v", "w", "x", "y", "z", "0", "1", "2", "3"
      , "4", "5", "6", "7", "8", "9"};

  /** wandelt den regulaeren Ausdruck in einen endlichen Automaten
   * um.
   * @return endlicher Automat, der Sprache akzeptiert, die vom
   *         regulaeren Ausdruck beschrieben wird
   */
  public abstract EndlicherAutomat alsAutomat();
  
  /** wandelt den uebergebenen String in einen regulaeren Ausdruck um
   * 
   * @param text umzuwandelnder String
   * @return resultierender Ausdruck
   */
  public static RegulaererAusdruck alsAusdruck(String text) {
    return alsAusdruck(text, false);
  }
  
  /**
   * wandelt den uebergebenen String in einen regulaeren Ausdruck um,
   * zweiter Parameter gibt an, ob ein Ableitungsbaum fuer den 
   * regulaeren Ausdruck gezeichnet werden soll
   * @param text umzuwandelnder String
   * @param graphik soll Ableitungsbaum gezeigt werden?
   * @returnresultierender Ausdruck
   */
  public static RegulaererAusdruck alsAusdruck(String text, boolean graphik) {
    kfg = new KontextfreieGrammatik();
    kfg.stringAlsGrammatik(grammatik);

    Ableitung abl = kfg.berechneAbleitung(kfg.stringAlsWort(text));
    if (!abl.isErfolgreich()) {
      //System.out.println("reg:" + kfg.getTerminale());
      throw new IllegalStateException(text + " nicht ableitbar, nur :" + abl.getAbgeleitet()); 
    }
    Ableitungsbaum abld = Ableitungsbaum.ableitungAlsBaum(abl);
    if(graphik) {
      abld.visualisieren();
    }
    return baumZuAusdruck(abld);
  }

  private static RegulaererAusdruck baumZuAusdruck(Ableitungsbaum ab) {
    //System.out.println("Regel: " + ab.getKfr() + "\n"+ab.get(0) + "\n" + ab);
    switch(ab.getKfr().getId()) {
      case "r1_0": return new Stern(baumZuAusdruck(ab.get(1)));
      case "r1_1": return new Oder(baumZuAusdruck(ab.get(1)), baumZuAusdruck(ab.get(3)));
      case "r1_2": return new Punkt(baumZuAusdruck(ab.get(0)), baumZuAusdruck(ab.get(1)));
      case "r1_3": return new Klammern(baumZuAusdruck(ab.get(1)));
      default: { 
        String id = ab.getKfr().getId().substring(3);
        return Terminal.terminal(fin[Integer.parseInt(id)]);    
      }
    }
  }
  
  /** Prueft ob der uebergebene String als Wort zur Sprache des
   * regulaeren Ausdrucks gehoert
   * @param text zu uberpruefender String
   * @return gehoert String zur Sprache des Ausdrucks?
   */
  public boolean beinhaltet(String text) {
    return this.beinhaltet(new Wort(text));
  }
  
  /** Prueft ob das uebergebene Wort zur Sprache des
   * regulaeren Ausdrucks gehoert
   * @param wort zu uberpruefendes Wort
   * @return gehoert Wort zur Sprache des Ausdrucks?
   */
  public boolean beinhaltet(Wort wort) {
    return this.alsAutomat().akzeptieren(wort,false);
  }
  
  /** Prueft, ob der als String uebergebene regulaere Ausdruck
   * sprachaequivalent mit this ist
   * @param reg zweiter Ausdruck
   * @return sind bei Ausdruecke sprachaequivalent
   */
  public boolean istAequivalent(String reg) {
    return this.istAequivalent(RegulaererAusdruck.alsAusdruck(reg));
  }
  
  /** Prueft, ob der uebergebene regulaere Ausdruck
   * sprachaequivalent mit this ist
   * @param reg zweiter Ausdruck
   * @return sind bei Ausdruecke sprachaequivalent
   */
  public boolean istAequivalent(RegulaererAusdruck re) {
    EndlicherAutomat ea1 = this.alsAutomat().deterministisch().minimieren();
    EndlicherAutomat ea2 = re.alsAutomat().deterministisch().minimieren();
    return ea1.istMinimalIsomorph(ea2);
  }
}
