package test.alphabet;

import java.util.List;

import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

import alphabet.Wort;
import alphabet.Zeichen;

public class WortTest {
  
  private Zeichen a;
  private Zeichen b;
  
  @BeforeEach
  public void setUp() {
    Zeichen.reset();
    this.a = Zeichen.zeichen("a");
    this.b = Zeichen.zeichen("b");
  }
  
  @Test
  public void testKonstruktorMitZeichen() {
    Assertions.assertEquals(new Wort(""), new Wort());
    Assertions.assertEquals(new Wort("aba"), new Wort(a, b, a));
  }
  
  @Test
  public void testleer() {
    Wort wort = new Wort("");
    Assertions.assertEquals(0, wort.laenge());
  }
  
  // hier wird bewusst mal von AAA abgewichen
  @Test
  public void testVorneUndAnfuegenUndLaenge() {
    Wort wort = new Wort("");
    Assertions.assertEquals(0, wort.laenge());
    
    wort.vorne(a);
    Assertions.assertEquals(1, wort.laenge());
    Assertions.assertEquals(a, wort.at(0));
    
    wort.anfuegen(b);
    Assertions.assertEquals(2, wort.laenge());
    Assertions.assertEquals(new Wort("ab"), wort);
    
    wort.vorne(a);
    Assertions.assertEquals(3, wort.laenge());
    Assertions.assertEquals(new Wort("aab"), wort);
    
    wort.anfuegen(b);
    Assertions.assertEquals(4, wort.laenge());
    Assertions.assertEquals(new Wort("aabb"), wort);
  }
  
  @Test
  public void testAt(){
    Wort w = new Wort("abba");
    Assertions.assertSame(a, w.at(0));
    Assertions.assertSame(b, w.at(1));
    Assertions.assertSame(b, w.at(2));
    Assertions.assertSame(a, w.at(3));
    try {
      w.at(4);
      Assertions.fail(w + " hat kein Zeichen an 4-ter Position");
    } catch (IllegalStateException e) {
    }
    try {
      w.at(-1);
      Assertions.fail(w + " hat kein Zeichen an -1-ter Position");
    } catch (IllegalStateException e) {
    }
  }
  
  @Test
  public void testErsetzeAnPositionMit() {
    Wort w = new Wort("abba");
    Zeichen.zeichen("c");
    w.ersetzeAnPositionMit(3, new Wort("cc"));
    w.ersetzeAnPositionMit(0, new Wort("cc"));
    Assertions.assertEquals(new Wort("ccbbcc"), w);
    try {
      w.ersetzeAnPositionMit(6, new Wort("cc"));
      Assertions.fail(w + " hat keine Position 6 zum Ersetzen");
    } catch (IllegalStateException e) {
    }
    try {
      w.ersetzeAnPositionMit(-1, new Wort("cc"));
      Assertions.fail(w + " hat keine Position 6 zum Ersetzen");
    } catch (IllegalStateException e) {
    }
  }
  
  @Test
  public void testLoeschen() {
    Wort w = new Wort("abba");
    w.loeschen(3);
    w.loeschen(0);
    Assertions.assertEquals(new Wort("bb"), w);
    try {
      w.loeschen(2);
      Assertions.fail(w + " hat keine Position 2 zum Loeschen");
    } catch (IllegalStateException e) {
    }
    try {
      w.loeschen(-1);
      Assertions.fail(w + " hat keine Position -1 zum Loeschen");
    } catch (IllegalStateException e) {
    }
  }
  
  @Test
  public void testPositionenVon() {
    Wort w = new Wort("");
    List<Integer> pos = w.positionenVon(a);
    Assertions.assertEquals(0, pos.size());
    
    w = new Wort("abba");
    pos = w.positionenVon(a);
    Assertions.assertEquals(2, pos.size());
    Assertions.assertTrue(pos.contains(0));
    Assertions.assertTrue(pos.contains(3));
    
    Zeichen c = Zeichen.zeichen("c");
    pos = w.positionenVon(c);
    Assertions.assertEquals(0, pos.size());
  }
  
  @Test
  public void testStringAlsWort() {
    Wort w = Wort.stringAlsWort("");
    Assertions.assertEquals(w, new Wort()); 
    
    w = Wort.stringAlsWort("aba");
    Assertions.assertEquals(w, new Wort(a, b, a));
    
    try {
      w = Wort.stringAlsWort("aca");
      Assertions.fail(w + " nicht vollstaendig in Zeichen uebersetzbar");
    } catch (IllegalStateException e) {
    }
  }
  
  @Test
  public void testToString() {
    Assertions.assertEquals("", new Wort().toString()); 
    Assertions.assertEquals("aba", new Wort("aba").toString()); 
    Zeichen.zeichen(" ");
    Assertions.assertEquals(" b b ", new Wort(" b b ").toString()); 
    Zeichen.zeichen("/space");
    Assertions.assertEquals(" b b ", new Wort("/spaceb/spaceb/space").toString()); 
  }
  
  @Test
  public void testToFile() {
    Assertions.assertEquals("", new Wort().toFile()); 
    Assertions.assertEquals("aba", new Wort("aba").toFile()); 
    Zeichen.zeichen(" ");
    Assertions.assertEquals("/spaceb/spaceb/space", new Wort(" b b ").toFile()); 
    Zeichen.zeichen("/space");
    Assertions.assertEquals("/spaceb/spaceb/space", new Wort("/spaceb/spaceb/space").toFile()); 
  }
  
  @Test
  public void testToGui() {
    Assertions.assertEquals("\u03B5", new Wort().toGui()); 
    Assertions.assertEquals("aba", new Wort("aba").toGui()); 
    Zeichen.zeichen(" ");
    Assertions.assertEquals(" b b ", new Wort(" b b ").toGui()); 
    Zeichen.zeichen("/space");
    Assertions.assertEquals(" b b ", new Wort("/spaceb/spaceb/space").toGui()); 
  }
  
  @Test
  public void testVonBis(){
    Wort w = new Wort("abba");
    Assertions.assertEquals(new Wort(""), w.vonBis(0,0));
    Assertions.assertEquals(new Wort("a"), w.vonBis(0,1));
    Assertions.assertEquals(new Wort("bb"), w.vonBis(1,3));
    Assertions.assertEquals(w, w.vonBis(0,4));
    try {
      w.vonBis(0,5);
      Assertions.fail(w + " hat kein Zeichen an 4-ter Position");
    } catch (IllegalStateException e) {
    }
    try {
      w.vonBis(-1,1);
      Assertions.fail(w + " hat kein Zeichen an -1-ter Position");
    } catch (IllegalStateException e) {
    }
  }
  
  @Test
  public void testIterator() {
    int i = 0;
    Wort w = new Wort(a, b, a);
    for(Zeichen z: w) {
      Assertions.assertEquals(z, w.at(i));
      i++;
    }
  }
  
  @Test
  public void testKomplexeZeichen(){
    Zeichen x1= Zeichen.zeichen("xa1");
    Zeichen x2= Zeichen.zeichen("xa2");
    Zeichen x3= Zeichen.zeichen("xa3");
    Zeichen x4= Zeichen.zeichen("xa4");
    
    Wort w = new Wort("xa1xa1xa2xa2");
    Assertions.assertEquals(w, new Wort(x1,x1,x2,x2));
    Assertions.assertEquals(4, w.laenge());
    Assertions.assertEquals(2, w.positionenVon(x2).size());
    Assertions.assertEquals(2, w.positionenVon(x1).size());
    Assertions.assertEquals(0, w.positionenVon(x3).size());
    Assertions.assertEquals("xa3xa4", w.ersetzeAnPositionMit(0
        , new Wort(x3)).ersetzeAnPositionMit(1, new Wort(x4))
                       .vonBis(0, 2).toString());
    
    
  }
  

}
