package test.endlicherAutomat;

import java.util.List;

import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

import alphabet.Nichtterminal;
import alphabet.Terminal;
import alphabet.Wort;
import alphabet.Zeichen;
import endlicherAutomat.AutomatUeberfuehrungsfunktion;
import endlicherAutomat.EndlicherAutomat;
import regulaererAusdruck.RegulaererAusdruck;
import zustand.Zustand;

public class AutomatTest {
  
  @BeforeEach
  public void setUp() throws Exception {
    Terminal.reset();
    Nichtterminal.reset();
    Zustand.reset();
    Zeichen.reset();
  }
  
  @Test
  public void testVervollstaendigen() {
    EndlicherAutomat ea = new EndlicherAutomat();
    ea.stringAlsAutomat("""
        Z: z1 z2 z3 z4
        E: z3
        S: z1
        A: a b 
        z1 a z2
        z1 b z2
        z2 a z3
        z2 a z4
        z2 b z3
        """);
    Assertions.assertFalse(ea.istVollstaendig());
    Assertions.assertEquals(2, ea.getAlphabet().size());
    Assertions.assertFalse(ea.getUeber()
        .istDefiniert(Zustand.zustand("z3"), Terminal.terminal("a")));
    Assertions.assertFalse(ea.getUeber()
        .istDefiniert(Zustand.zustand("z3"), Terminal.terminal("b")));
    ea.vervollstaendigen();
    Assertions.assertEquals(2, ea.getAlphabet().size());
    //System.out.println("vollstaendig:\n" + ea);
    Assertions.assertTrue(ea.getUeber()
        .istDefiniert(Zustand.zustand("z3"), Terminal.terminal("a")));
    Assertions.assertTrue(ea.getUeber()
        .istDefiniert(Zustand.zustand("z3"), Terminal.terminal("b")));
    Assertions.assertTrue(ea.istVollstaendig(), ea + "");
  }
  
  @Test
  public void testOhneEpsilon() {
    EndlicherAutomat ea = new EndlicherAutomat();
    ea.stringAlsAutomat("""
        Z: z1 z2 z3 z4
        E: z3
        S: z1
        A: a b 
        z1 /eps z2
        z1 /eps z1
        z3 /eps z4
        z2 a z3
        z2 a z4
        z2 b z3
        """);
    Assertions.assertFalse(ea.istOhneEpsilon());
    Assertions.assertEquals(2, ea.getAlphabet().size());
    ea.epsilonEntfernen();
    Assertions.assertEquals(2, ea.getAlphabet().size());
    //System.out.println("vollstaendig:\n" + ea);
    Assertions.assertTrue(ea.getUeber()
        .istDefiniert(Zustand.zustand("z1"), Terminal.terminal("a")));
    Assertions.assertTrue(ea.getUeber()
        .istDefiniert(Zustand.zustand("z1"), Terminal.terminal("b")));
    Assertions.assertTrue(ea.getUeber()
        .istDefiniert(Zustand.zustand("z2"), Terminal.terminal("b")));
    Assertions.assertTrue(ea.getUeber()
        .moeglicheFolgezustaende(Zustand.zustand("z2"), Terminal.terminal("b"))
        .contains(Zustand.zustand("z4")));
    Assertions.assertTrue(ea.istOhneEpsilon(), ea + "");
  }
  
  @Test
  public void testDeterministisch() {
    EndlicherAutomat ea = new EndlicherAutomat();
    ea.stringAlsAutomat("""
        Z: z1 z2 z3 z4
        E: z3
        S: z1
        A: a b 
        z1 /eps z2
        z1 /eps z1
        z3 /eps z4
        z2 a z3
        z2 a z4
        z2 b z3
        """);
    Assertions.assertEquals(2, ea.getAlphabet().size());
    ea.epsilonEntfernen();
    ea.vervollstaendigen();
    Assertions.assertEquals(2, ea.getAlphabet().size());
    Assertions.assertFalse(ea.istDeterministisch());
    ea = ea.deterministisch();
    Assertions.assertEquals(2, ea.getAlphabet().size());
    Assertions.assertTrue(ea.istDeterministisch(), ea + "");
  }
  
  @Test
  public void testEpsilonNutzen() {
    EndlicherAutomat ea = new EndlicherAutomat();
    ea.stringAlsAutomat("""
        Z: z1 z2 z3 z4
        E: z3
        S: z1
        A: a b 
        z1 /eps z2
        z1 /eps z1
        z3 /eps z4
        z2 a z3
        z2 a z4
        z2 b z3
        """);
    //ea.vervollstaendigen();
    Assertions.assertEquals(2, ea.getAlphabet().size());
    Assertions.assertTrue(ea.akzeptieren("a"));
    Assertions.assertTrue(ea.akzeptieren("b"));
    Assertions.assertFalse(ea.akzeptieren(""));
    Assertions.assertFalse(ea.akzeptieren("aa"));
    Assertions.assertFalse(ea.akzeptieren("ab"));
  }

  @Test
  public void testXoderY() {
    EndlicherAutomat ea = new EndlicherAutomat();
    ea.stringAlsAutomat("""
        Z: z22 z23 z24 z25 z26 z27
        E: z23 z25 z27
        S: z22
        A: x y 
        z26 y z27
        z24 x z25
        z25 /eps z23
        z22 /eps z24
        z22 /eps z26
        z27 /eps z23
        """);
    //ea.vervollstaendigen();
    Assertions.assertEquals(2, ea.getAlphabet().size());
    Assertions.assertFalse(ea.istDeterministisch());
    Assertions.assertFalse(ea.istOhneEpsilon());
    //System.out.println(ea);
    
    //ea.epsilonEntfernen();
    //System.out.println("--------\n" + ea);
    
    //ea.deterministisch();
    //Assertions.assertTrue(ea.istOhneEpsilon());
    //System.out.println("--------det\n" + ea);
    //Assertions.assertTrue(ea.akzeptieren("x"), ea.toString() + "\n" + ea.getUeber().folgeZustand(Zustand.getZustand("z22"), Zeichen.getZeichen("x")));
    Assertions.assertTrue(ea.akzeptieren("x"), ea.toString());
    Assertions.assertTrue(ea.akzeptieren("y"), ea.toString());
    Assertions.assertFalse(ea.akzeptieren(""));
    Assertions.assertFalse(ea.akzeptieren("xx"));
    Assertions.assertFalse(ea.akzeptieren("xy"));
  }
  
  // ist (x+y)*
  //          \/------------------e-
  //   7 -e-> 1 -e-> 3 -x-> 4 -e->
  //            -e-> 5 -y-> 6 -e-> 2
  //     -e-> 8 <-----------------e-   
  
  @Test
  public void testAutomatUeberfuehrung() {
    AutomatUeberfuehrungsfunktion sut = new AutomatUeberfuehrungsfunktion();
    Zustand z1 = Zustand.zustand("z1");
    Zustand z2 = Zustand.zustand("z2");
    Zustand z3 = Zustand.zustand("z3");
    Zustand z4 = Zustand.zustand("z4");
    Zustand z5 = Zustand.zustand("z5");
    Zustand z6 = Zustand.zustand("z6");
    Zustand z7 = Zustand.zustand("z7");
    Zustand z8 = Zustand.zustand("z8");
    Terminal x = Terminal.terminal("x");
    Terminal y = Terminal.terminal("y");
    sut.add(z3, x, z4);
    sut.add(z5, y, z6);
    sut.addEpsilon(z7, z1);
    sut.addEpsilon(z7, z8);
    sut.addEpsilon(z1, z3);
    sut.addEpsilon(z1, z5);
    sut.addEpsilon(z4, z2);
    sut.addEpsilon(z6, z2);
    sut.addEpsilon(z2, z8);
    sut.addEpsilon(z2, z1);
    
    Assertions.assertTrue(sut.istDefiniert(z3, x));
    Assertions.assertFalse(sut.istDefiniert(z3, y));
    Assertions.assertFalse(sut.istDefiniert(z2, y));
    
    Assertions.assertTrue(sut.istEindeutig(z3, x));
    Assertions.assertFalse(sut.istEindeutig(z3, y));
    Assertions.assertFalse(sut.istEindeutig(z2, y));
    
    List<Zustand> meps = sut.mitEpsilonErreichbar(z7);
    Assertions.assertEquals(5, meps.size(), z7 + ": " + meps);
    meps = sut.mitEpsilonErreichbar(z1);
    Assertions.assertEquals(3, meps.size(), z1 + ": " + meps);
    meps = sut.mitEpsilonErreichbar(z5);
    Assertions.assertEquals(1, meps.size(), z5 + ": " + meps);
    meps = sut.mitEpsilonErreichbar(z4);
    Assertions.assertEquals(6, meps.size(), z4 + ": " + meps);
    meps = sut.mitEpsilonErreichbar(z6);
    Assertions.assertEquals(6, meps.size(), z5 + ": " + meps);
    meps = sut.mitEpsilonErreichbar(z2);
    Assertions.assertEquals(5, meps.size(), z2 + ": " + meps);
    
    Zustand[] erreichbar = {z1, z3, z5, z4, z2, z8};
    for(Zustand z:erreichbar) {
        if(z != z5 && z != z8) {
        List<Zustand> next = sut.moeglicheFolgezustaende(z, x);
        Assertions.assertEquals(6, next.size(), z + ": " + next);
        for(Zustand zu:erreichbar) {
          Assertions.assertTrue(next.contains(zu));
        }
      }
    }
  }
  
  @Test
  public void testEpsilonEntfernenGenauer() {
    EndlicherAutomat ea = new EndlicherAutomat();
    ea.stringAlsAutomat("""
        Z: z1 z2 z3 z4 z5 z6 z7 z8
        E: z8
        S: z7
        A: x y 
        z5 y z6
        z3 x z4
        z7 /eps z1
        z7 /eps z8
        z2 /eps z8
        z2 /eps z1
        z1 /eps z3
        z1 /eps z5
        z4 /eps z2
        z6 /eps z2
        """);
    //ea.vervollstaendigen();
    Assertions.assertEquals(2, ea.getAlphabet().size());
    Assertions.assertFalse(ea.istDeterministisch());
    Assertions.assertFalse(ea.istOhneEpsilon());
    // System.out.println(ea);
    
    
    
    // System.out.println(ea.getUeber().moeglicheFolgezustaende(Zustand.zustand("z5"), Terminal.terminal("y")));
    // System.out.println(ea.getUeber().mitEpsilonErreichbar(Zustand.zustand("z6")));
    // System.out.println(ea.getUeber().getEpsilon().get(Zustand.zustand("z6")));
    ea.epsilonEntfernen();    
    // System.out.println("--------ohneEps\n" + ea);
    Zustand[] zyklus = {Zustand.zustand("z1"), Zustand.zustand("z2")
        //, Zustand.zustand("z3"),  Zustand.zustand("z4")
        , Zustand.zustand("z5"), Zustand.zustand("z6")};
    for(Zustand z1: zyklus) {
      List<Zustand> err = ea.getUeber().moeglicheFolgezustaende(z1, Terminal.terminal("y"));
      Assertions.assertEquals(6, err.size(), z1 + ": " + err +"\n" + ea);
      for(Zustand z2: zyklus) {
        Assertions.assertTrue (err.contains(z2), z2 + " nicht mit y nach " + z1);
      }
    }
  }
  
  @Test
  public void testDeterministischerAutomat() {
    EndlicherAutomat ea = new EndlicherAutomat();
    ea.stringAlsAutomat("""
        % Worte, die mit x enden: (x+y)*x
        Z: z1 z2
        E: z2
        S: z1
        A: x y 
        z1 x z2
        z1 y z1
        z2 x z2
        z2 y z1
        """);
    Assertions.assertEquals(2, ea.getAlphabet().size());
    Assertions.assertTrue(ea.istDeterministisch());
    Assertions.assertTrue(ea.istOhneEpsilon());
    Assertions.assertTrue(ea.istVollstaendig());
    Assertions.assertEquals(4, ea.getUeber().getFunktion().size());
    Assertions.assertEquals(0, ea.getUeber().getEpsilon().size());
    
    List<Zustand> z1 = ea.getUeber().moeglicheFolgezustaende(
        Zustand.zustand("z1"), Terminal.terminal("x"));
    Assertions.assertEquals(1, z1.size());
    Assertions.assertTrue(z1.contains(Zustand.zustand("z2")));
    
    ea.epsilonEntfernen();
    Assertions.assertEquals(2, ea.getAlphabet().size());
    Assertions.assertTrue(ea.istDeterministisch());
    Assertions.assertTrue(ea.istOhneEpsilon());
    Assertions.assertTrue(ea.istVollstaendig());
    Assertions.assertEquals(4, ea.getUeber().getFunktion().size());
    Assertions.assertEquals(0, ea.getUeber().getEpsilon().size());
    
    ea.vervollstaendigen();
    
    Assertions.assertEquals(2, ea.getAlphabet().size());
    Assertions.assertTrue(ea.istDeterministisch());
    Assertions.assertTrue(ea.istOhneEpsilon());
    Assertions.assertTrue(ea.istVollstaendig());
    Assertions.assertEquals(4, ea.getUeber().getFunktion().size());
    Assertions.assertEquals(0, ea.getUeber().getEpsilon().size());
    
    ea.deterministisch();
    Assertions.assertEquals(2, ea.getAlphabet().size());
    Assertions.assertTrue(ea.istDeterministisch());
    Assertions.assertTrue(ea.istOhneEpsilon());
    Assertions.assertTrue(ea.istVollstaendig());
    Assertions.assertEquals(4, ea.getUeber().getFunktion().size());
    Assertions.assertEquals(0, ea.getUeber().getEpsilon().size());
    
    ea.minimieren();
    Assertions.assertEquals(2, ea.getAlphabet().size());
    Assertions.assertTrue(ea.istDeterministisch());
    Assertions.assertTrue(ea.istOhneEpsilon());
    Assertions.assertTrue(ea.istVollstaendig());
    Assertions.assertEquals(4, ea.getUeber().getFunktion().size());
    Assertions.assertEquals(0, ea.getUeber().getEpsilon().size());
    
    Assertions.assertTrue(ea.akzeptieren("x"));
    Assertions.assertTrue(ea.akzeptieren("yxyyyxyyxyxyxxxxx"));
    Assertions.assertFalse(ea.akzeptieren(""));
    Assertions.assertFalse(ea.akzeptieren("xxxxy"));
    
  }
  
  @Test
  public void testSchleifeUmXoderY() {
    EndlicherAutomat ea = new EndlicherAutomat();
    ea.stringAlsAutomat("""
        Z: z21 z22 z23 z24 z25 z26 z27 z28
        E: z28
        S: z27
        A: x y 
        z25 y z26
        z23 x z24
        z27 /eps z21
        z27 /eps z28
        z22 /eps z28
        z22 /eps z21
        z21 /eps z23
        z21 /eps z25
        z24 /eps z22
        z26 /eps z22
        """);
    //ea.vervollstaendigen();
    Assertions.assertEquals(2, ea.getAlphabet().size());
    Assertions.assertFalse(ea.istDeterministisch());
    Assertions.assertFalse(ea.istOhneEpsilon());
    // System.out.println(ea);
    
    ea.epsilonEntfernen();
    //System.out.println("--------ohneEps\n" + ea);
    
    //ea.deterministisch();
    //Assertions.assertTrue(ea.istOhneEpsilon());
    //System.out.println("--------det\n" + ea);
    //Assertions.assertTrue(ea.akzeptieren("x"), ea.toString() + "\n" + ea.getUeber().folgeZustand(Zustand.getZustand("z22"), Zeichen.getZeichen("x")));
    Assertions.assertTrue(ea.akzeptieren("x"), ea.toString());
    Assertions.assertTrue(ea.akzeptieren("y"), ea.toString());
    Assertions.assertTrue(ea.akzeptieren(""));
    Assertions.assertTrue(ea.akzeptieren("xx"));
    Assertions.assertTrue(ea.akzeptieren("xy"));
  }
  
  @Test
  public void testMinimieren() {
    EndlicherAutomat ea = new EndlicherAutomat();
    ea.stringAlsAutomat("""
        Z: z1 z2 z3 z4 z5 z6
        E: z1 z3 z5
        S: z1
        A: x y 
        z1 x z2
        z2 x z3
        z3 x z4
        z4 x z5
        z5 x z6
        z6 x z1 
        """);
    Assertions.assertEquals(2, ea.getAlphabet().size());
    ea.vervollstaendigen();
    Assertions.assertEquals(2, ea.getAlphabet().size());
    Assertions.assertTrue(ea.istDeterministisch());
    Assertions.assertTrue(ea.istOhneEpsilon());
    
    ea.minimieren();
    Assertions.assertEquals(2, ea.getAlphabet().size());
    Assertions.assertEquals(3, ea.getZustaende().size(), "Zustaende: " + ea.getZustaende());
    Assertions.assertEquals(1, ea.getEndzustaende().size(), "Endzustaende: " + ea.getEndzustaende());
    Assertions.assertTrue(ea.akzeptieren("xx"), ea.toString());
    Assertions.assertTrue(ea.akzeptieren("xxxx"), ea.toString());
    Assertions.assertTrue(ea.akzeptieren(""));
    Assertions.assertFalse(ea.akzeptieren("xy"));
    Assertions.assertFalse(ea.akzeptieren("x"));
  }
  
  @Test
  public void testAlsRegulaererAusdruck0() {
    EndlicherAutomat ea = new EndlicherAutomat();
    ea.stringAlsAutomat("""
        Z: z1 z2
        E: z2
        S: z1
        A: a b 
        z1 a z1
        z1 b z1
        z2 a z2
        z2 b z1
        """);
    RegulaererAusdruck re = ea.alsRegulaererAusdruck();
    Assertions.assertEquals("{}", re.toString());
  }
  
  // folgende Tests nicht weitergeben, da die Reihenfolge a+b oder b+a
  // immer unterschiedlich sein kann
  @Test
  public void testAlsRegulaererAusruck1() {
    EndlicherAutomat ea = new EndlicherAutomat(); //EGH
    ea.stringAlsAutomat("""
        Z: z1 z2
        E: z2
        S: z1
        A: a b 
        z1 a z1
        z1 b z2
        z2 a z2
        z2 b z1
        """);
    RegulaererAusdruck re = ea.alsRegulaererAusdruck();
    Assertions.assertEquals("((a + b(a)*b))*b(a)*", re.toString());
  }
  
  @Test
  public void testAlsRegulaererAusruck2() { 
    EndlicherAutomat ea = new EndlicherAutomat(); // GH
    ea.stringAlsAutomat("""
        Z: z1 z2
        E: z2
        S: z1
        A: a b 
        z1 a z2
        z1 b z2
        z2 a z2
        z2 b z1
        """);
    RegulaererAusdruck re = ea.alsRegulaererAusdruck();
    Assertions.assertEquals("(((a + b)(a)*b))*(a + b)(a)*", re.toString());
  }
  
  @Test
  public void testAlsRegulaererAusruck3() {
    EndlicherAutomat ea = new EndlicherAutomat(); // H
    ea.stringAlsAutomat("""
        Z: z1 z2
        E: z2
        S: z1
        A: a b 
        z1 a z2
        z1 b z2
        z2 a z2
        z2 b z2
        """);
    RegulaererAusdruck re = ea.alsRegulaererAusdruck();
    Assertions.assertTrue("(a + b)((a + b))*".equals(re.toString())
        || "(a + b)((b + a))*".equals(re.toString()));
  }
  
  @Test
  public void testAlsRegulaererAusruck4() {
    EndlicherAutomat ea = new EndlicherAutomat(); // G
    ea.stringAlsAutomat("""
        Z: z1 z2
        E: z2
        S: z1
        A: a b 
        z1 a z2
        z1 b z2
        z2 a z1
        z2 b z1
        """);
    RegulaererAusdruck re = ea.alsRegulaererAusdruck();
    Assertions.assertEquals("(((a + b)(a + b)))*(a + b)", re.toString());
  }
  
  @Test
  public void testAlsRegulaererAusruck5() {
    EndlicherAutomat ea = new EndlicherAutomat(); // EG
    ea.stringAlsAutomat("""
        Z: z1 z2
        E: z2
        S: z1
        A: a b 
        z1 a z1
        z1 b z2
        z2 a z1
        z2 b z1
        """);
    RegulaererAusdruck re = ea.alsRegulaererAusdruck();
    Assertions.assertTrue("((a + b(a + b)))*b".equals(re.toString())
        || "((a + b(b + a)))*b".equals(re.toString()));
  }
  
  @Test
  public void testAlsRegulaererAusruck6() {
    EndlicherAutomat ea = new EndlicherAutomat(); // EH
    ea.stringAlsAutomat("""
        Z: z1 z2
        E: z2
        S: z1
        A: a b 
        z1 a z1
        z1 b z2
        z2 a z2
        z2 b z2
        """);
    RegulaererAusdruck re = ea.alsRegulaererAusdruck();
    Assertions.assertEquals("(a)*b((a + b))*", re.toString());
  }
  
  @Test
  public void testAlsRegulaererAusruck7() { 
    EndlicherAutomat ea = new EndlicherAutomat(); //E
    ea.stringAlsAutomat("""
        Z: z1 z2
        E: z2
        S: z1
        A: a b 
        z1 a z1
        z1 b z2
        """);
    RegulaererAusdruck re = ea.alsRegulaererAusdruck();
    Assertions.assertEquals("(a)*b", re.toString());
  }
  
  @Test
  public void testAlsRegulaererAusruck8() {
    EndlicherAutomat ea = new EndlicherAutomat();
    ea.stringAlsAutomat("""
        Z: z1 z2
        E: z2 z1
        S: z1
        A: a b 
        z1 a z1
        z1 b z2
        z2 a z2
        """);
    RegulaererAusdruck re = ea.alsRegulaererAusdruck();
    Assertions.assertEquals("((a)* + (a)*b(a)*)", re.toString());
  }
  
  @Test
  public void testAlsRegulaererAusruck9() {
    EndlicherAutomat ea = new EndlicherAutomat();
    ea.stringAlsAutomat("""
        Z: z1 z2 z3
        E: z2 z1 z3
        S: z1
        A: a b 
        z1 a z2
        z2 b z3
        """);
    RegulaererAusdruck re = ea.alsRegulaererAusdruck();
    Assertions.assertEquals("((({})* + a) + ab)", re.toString());
  }
  
  @Test
  public void testAlsRegulaererAusruck10() {
    EndlicherAutomat ea = new EndlicherAutomat();
    ea.stringAlsAutomat("""
        Z: z1 z2 z3
        E: z2 z1 z3
        S: z1
        A: a b c
        z1 a z2
        z1 b z1
        z1 c z1
        z2 b z3
        """);
    RegulaererAusdruck re = ea.alsRegulaererAusdruck();
    Assertions.assertTrue("((((b + c))* + ((b + c))*a) + ((b + c))*ab)".equals(re.toString())
        || "((((c + b))* + ((c + b))*a) + ((c + b))*ab)".equals(re.toString()) );
  }
  
  @Test
  public void testAlsRegulaererAusruck11() {
    EndlicherAutomat ea = new EndlicherAutomat();
    ea.stringAlsAutomat("""
        Z: z0 z1 z2 z3 z4
        E: z3 z4
        S: z0
        A: a b 
        z0 a z0
        z0 b z3
        z1 a z1
        z1 b z1
        z2 a z4
        z2 b z0
        z3 a z2
        z3 b z4
        z4 a z1
        z4 b z1
        """);
    RegulaererAusdruck re = ea.alsRegulaererAusdruck();
    
    EndlicherAutomat ea2 = re.alsAutomat();
    Assertions.assertTrue(ea2.akzeptieren("b"));
    Assertions.assertTrue(ea2.akzeptieren("aab"));
    Assertions.assertTrue(ea2.akzeptieren("bb"));
    Assertions.assertTrue(ea2.akzeptieren("babb"));
    Assertions.assertTrue(ea2.akzeptieren("babbabb"));
    Assertions.assertTrue(ea2.akzeptieren("baa"));
    Assertions.assertTrue(re.toString().equals("(((a + bab))*b + ((a + bab))*b(aa + b))")
        || re.toString().equals("(((a + bab))*b + ((a + bab))*b(b + aa))")
        || re.toString().equals("(((a + bab))*b + ((bab + a))*b(b + aa))")
        || re.toString().equals("(((a + bab))*b + ((bab + a))*b(aa + b))"), re.toString());
  }
  
  @Test
  public void testAlsRegulaererAusruck12() {
    EndlicherAutomat ea = new EndlicherAutomat();
    ea.stringAlsAutomat("""
        Z: z0 z1 z2 z3 z4
        E: z3 z4
        S: z0
        A: a b 
        z0 a z0
        z0 b z4
        z1 a z1
        z1 b z1
        z2 a z3
        z2 b z0
        z4 a z2
        z4 b z3
        z3 a z1
        z3 b z1
        """);
    RegulaererAusdruck re = ea.alsRegulaererAusdruck();
    
    EndlicherAutomat ea2 = re.alsAutomat();
    Assertions.assertTrue(ea2.akzeptieren("b"));
    Assertions.assertTrue(ea2.akzeptieren("aab"));
    Assertions.assertTrue(ea2.akzeptieren("bb"));
    Assertions.assertTrue(ea2.akzeptieren("babb"));
    Assertions.assertTrue(ea2.akzeptieren("babbabb"));
    Assertions.assertTrue(ea2.akzeptieren("baa"));
    Assertions.assertTrue(
             "(((a + bab))*b(b + aa) + ((a + bab))*b)".equals(re.toString())
          || "(((a + bab))*b(aa + b) + ((a + bab))*b)".equals(re.toString())
          || "(((a + bab))*b(aa + b) + ((bab + a))*b)".equals(re.toString())    
          || "(((a + bab))*b(b + aa) + ((bab + a))*b)".equals(re.toString()) 
          || "(((bab + a))*b(b + aa) + ((a + bab))*b)".equals(re.toString())
          || "(((bab + a))*b(aa + b) + ((a + bab))*b)".equals(re.toString())
          || "(((bab + a))*b(aa + b) + ((bab + a))*b)".equals(re.toString())    
          || "(((bab + a))*b(b + aa) + ((bab + a))*b)".equals(re.toString()) 
        , re.toString());
  }
  
  @Test
  public void testUmbenennen() {
    EndlicherAutomat ea = new EndlicherAutomat();
    ea.stringAlsAutomat("""
        % Worte, die mit x enden: (x+y)*x
        Z: z1 z2
        E: z2
        S: z1
        A: x y 
        z1 x z2
        z1 y z1
        z2 x z2
        z2 y z1
        """);
    Assertions.assertEquals(2, ea.getAlphabet().size());
    ea.zustandsnamenKuerzen();
    Zustand z1 = ea.getStart();
    Zustand z2 = ea.getZustaende().get(1);
    Terminal x = Terminal.terminal("x");
    Terminal y = Terminal.terminal("y");
    Assertions.assertEquals(2, ea.getAlphabet().size());
    Assertions.assertEquals(2, ea.getZustaende().size());
    Assertions.assertEquals(z2, ea.getUeber().moeglicheFolgezustaende(z1, x).get(0));
    Assertions.assertEquals(z1, ea.getUeber().moeglicheFolgezustaende(z1, y).get(0));
    Assertions.assertEquals(z2, ea.getUeber().moeglicheFolgezustaende(z2, x).get(0));
    Assertions.assertEquals(z1, ea.getUeber().moeglicheFolgezustaende(z2, y).get(0));
  }
  
  @Test
  public void testMinimieren2(){
    EndlicherAutomat ea = new EndlicherAutomat();
    ea.stringAlsAutomat("""
        A: 10 20 50
        Z: 0 10 20 30 40 50 60
        E: 60
        S: 0
         0 10 10
        10 10 20
        20 10 30
        30 10 40
        40 10 50
        50 10 60
         0 20 20
        10 20 30
        20 20 40
        30 20 50
        40 20 60
        50 20 0
        0  50 50
        10 50 60
        20 50 0
        30 50 0
        40 50 0
        50 50 0
        """);
    // System.out.println(ea);
    Wort w = new Wort("102020505010");
    Assertions.assertTrue(ea.akzeptieren(w, false));
    Assertions.assertTrue(ea.deterministisch().akzeptieren(w, false));
    Assertions.assertTrue(ea.minimieren().akzeptieren(w, false));
  }
  
  @Test
  public void testMinimieren3(){
    EndlicherAutomat ea = new EndlicherAutomat();
    ea.stringAlsAutomat("""
        A: 10 20
        Z: 0 10 20 30
        E: 30
        S: 0
         0 10 10
        10 10 20
        20 10 30
         0 20 20
        10 20 30
        20 20 0
        """);
    // System.out.println(ea);
    Wort w = new Wort("20202010");
    Assertions.assertTrue(ea.akzeptieren(w, false));
    Assertions.assertTrue(ea.deterministisch().akzeptieren(w, false));
    Assertions.assertTrue(ea.minimieren().akzeptieren(w, false));
  }
  
  @Test
  public void testMinimieren4(){
    EndlicherAutomat ea = new EndlicherAutomat();
    ea.stringAlsAutomat("""
        A: 10 20
        Z: 0 10 20 30 40
        E: 40
        S: 0
         0 10 10
        10 10 20
        20 10 30
        30 10 40
         0 20 20
        10 20 30
        20 20 40
        30 20 0
        """);
    //System.out.println(ea);
    Wort w = new Wort("102020201010");
    Assertions.assertTrue(ea.akzeptieren(w, false));
    Assertions.assertTrue(ea.deterministisch().akzeptieren(w, false));
    Assertions.assertTrue(ea.minimieren().akzeptieren(w, false));
  }
  
  @Test
  public void testIstIsomorph1() {
    EndlicherAutomat ea = new EndlicherAutomat();
    ea.stringAlsAutomat("""
        Z: z1 z2
        E: z2
        S: z1
        A: a b 
        z1 a z2
        z1 b z2
        z2 a z2
        z2 b z2
        """);
    EndlicherAutomat ea2 = new EndlicherAutomat();
    ea2.stringAlsAutomat("""
        Z: z11 z12
        E: z12
        S: z11
        A: a b 
        z11 a z12
        z11 b z12
        z12 a z12
        z12 b z12
        """);
    
    Assertions.assertTrue(ea.istSprachaequivalent(ea2));
    Assertions.assertTrue(ea.istSprachaequivalent(ea));
    Assertions.assertTrue(ea2.istSprachaequivalent(ea));
    Assertions.assertTrue(ea2.istSprachaequivalent(ea2));
  }
  
  @Test
  public void testIstIsomorph2() {
    EndlicherAutomat ea = new EndlicherAutomat();
    ea.stringAlsAutomat("""
        A: a b
        Z: z0 z1 z2 z3 z4
        E: z4
        S: z0
        z0 a z1
        z1 /eps z2
        z1 a z2
        z2 /eps z3
        z2 b z4
        z3 /eps z0
        z3 /eps z4
        """);
    EndlicherAutomat ea2 = RegulaererAusdruck
        .alsAusdruck("(a(a)*b + a(a)*)").alsAutomat();
    
    Assertions.assertTrue(ea.istSprachaequivalent(ea2));
    Assertions.assertTrue(ea.istSprachaequivalent(ea));
    Assertions.assertTrue(ea2.istSprachaequivalent(ea));
    Assertions.assertTrue(ea2.istSprachaequivalent(ea2));
  }
  
  @Test
  public void testIstIsomorph3() {
    EndlicherAutomat ea = new EndlicherAutomat();
    ea.stringAlsAutomat("""
        A: a b
        Z: z0 z1 z2 z3 z4
        E: z4
        S: z0
        z0 a z1
        z1 /eps z2
        z1 a z2
        z2 /eps z3
        z2 b z4
        z3 /eps z0
        z3 /eps z4
        """);
    EndlicherAutomat ea2 = RegulaererAusdruck
        .alsAusdruck("(a(a)*b + a(a)*b)").alsAutomat();
    
    Assertions.assertFalse(ea.istSprachaequivalent(ea2));
    Assertions.assertFalse(ea2.istSprachaequivalent(ea));

  }
  
  @Test
  public void testIstIsomorph4() {
    EndlicherAutomat ea = new EndlicherAutomat();
    ea.stringAlsAutomat("""
        A: a b
        Z: z0 z1 z2 z3 z4
        E: z4
        S: z0
        z0 a z1
        z1 /eps z2
        z1 a z2
        z2 /eps z3
        z2 b z4
        z3 /eps z0
        z3 /eps z4
        """);
    EndlicherAutomat ea2 = RegulaererAusdruck
        .alsAusdruck("(a(a)*b + a(a)*c)").alsAutomat();
    
    try {
      ea.istSprachaequivalent(ea2);
    } catch (IllegalStateException e) {
    }
    try {
      ea2.istSprachaequivalent(ea);
    } catch (IllegalStateException e) {
    }
  }
  
  @Test
  public void testIstMinimalIsomorph1() {
    EndlicherAutomat ea = new EndlicherAutomat();
    ea.stringAlsAutomat("""
        Z: z1 z2
        E: z2
        S: z1
        A: a b 
        z1 a z2
        z1 b z2
        z2 a z2
        z2 b z2
        """);
    EndlicherAutomat ea2 = new EndlicherAutomat();
    ea2.stringAlsAutomat("""
        Z: z11 z12
        E: z12
        S: z11
        A: a b 
        z11 a z12
        z11 b z12
        z12 a z12
        z12 b z12
        """);
    
    Assertions.assertTrue(ea.istMinimalIsomorph(ea2));
    Assertions.assertTrue(ea.istMinimalIsomorph(ea));
    Assertions.assertTrue(ea2.istMinimalIsomorph(ea));
    Assertions.assertTrue(ea2.istMinimalIsomorph(ea2));
  }
  
  @Test
  public void testIstMinimalIsomorph2() {
    EndlicherAutomat ea = new EndlicherAutomat();
    ea.stringAlsAutomat("""
        Z: z1 z2
        E: z2
        S: z1
        A: a 
        z1 a z2
        //z1 b z1
        z2 a z1
        //z2 b z2
        """);
    EndlicherAutomat ea2 = RegulaererAusdruck.alsAusdruck("(aa)*a").alsAutomat().deterministisch().minimieren();  
    Assertions.assertTrue(ea.istMinimalIsomorph(ea2), ea2.getAlphabet().get(0) + "\n" + ea2.toString());
    Assertions.assertTrue(ea.istMinimalIsomorph(ea));
    Assertions.assertTrue(ea2.istMinimalIsomorph(ea));
    Assertions.assertTrue(ea2.istMinimalIsomorph(ea2));
    
    ea2 = RegulaererAusdruck.alsAusdruck("a(aa)*").alsAutomat().deterministisch().minimieren();  
    Assertions.assertTrue(ea.istMinimalIsomorph(ea2), ea2.getAlphabet().get(0) + "\n" + ea2.toString());
    Assertions.assertTrue(ea.istMinimalIsomorph(ea));
    Assertions.assertTrue(ea2.istMinimalIsomorph(ea));
    Assertions.assertTrue(ea2.istMinimalIsomorph(ea2));
    
    ea2 = RegulaererAusdruck.alsAusdruck("((aa)*aaa + a)").alsAutomat().deterministisch().minimieren();  
    Assertions.assertTrue(ea.istMinimalIsomorph(ea2), ea2.getAlphabet().get(0) + "\n" + ea2.toString());
    Assertions.assertTrue(ea.istMinimalIsomorph(ea));
    Assertions.assertTrue(ea2.istMinimalIsomorph(ea));
    Assertions.assertTrue(ea2.istMinimalIsomorph(ea2));
  }
  
  @Test
  public void testIstMinimalIsomorph3() {
    EndlicherAutomat ea = new EndlicherAutomat();
    ea.stringAlsAutomat("""
        Z: z1 z2
        E: z2
        S: z1
        A: a b
        z1 a z2
        z1 b z1
        z2 a z1
        z2 b z2
        """);
    // a ersetzen mit b*ab*
    EndlicherAutomat ea2 = RegulaererAusdruck.alsAusdruck("((b)*a(b)*(b)*a(b)*)*(b)*a(b)*", true).alsAutomat().deterministisch().minimieren();  
    Assertions.assertTrue(ea.istMinimalIsomorph(ea2), ea2.getAlphabet().get(0) + "\n" + ea2.toString());
    Assertions.assertTrue(ea.istMinimalIsomorph(ea));
    Assertions.assertTrue(ea2.istMinimalIsomorph(ea));
    Assertions.assertTrue(ea2.istMinimalIsomorph(ea2));
    
    ea2 = RegulaererAusdruck.alsAusdruck("(b)*a(b)*((b)*a(b)*(b)*a(b)*)*").alsAutomat().deterministisch().minimieren();  
    Assertions.assertTrue(ea.istMinimalIsomorph(ea2), ea2.getAlphabet().get(0) + "\n" + ea2.toString());
    Assertions.assertTrue(ea.istMinimalIsomorph(ea));
    Assertions.assertTrue(ea2.istMinimalIsomorph(ea));
    Assertions.assertTrue(ea2.istMinimalIsomorph(ea2));
    
//    ea2 = RegulaererAusdruck.alsAusdruck("(((b)*a(b)*(b)*a(b)*)*(b)*a(b)*(b)*a(b)*(b)*a(b)* + (b)*a(b)*)").alsAutomat().deterministisch().minimieren();  
//    Assertions.assertTrue(ea.istMinimalIsomorph(ea2), ea2.getAlphabet().get(0) + "\n" + ea2.toString());
//    Assertions.assertTrue(ea.istMinimalIsomorph(ea));
//    Assertions.assertTrue(ea2.istMinimalIsomorph(ea));
//    Assertions.assertTrue(ea2.istMinimalIsomorph(ea2));
  }
  
  @Test
  public void testIstMinimalIsomorph3negativ() {
    EndlicherAutomat ea = new EndlicherAutomat();
    ea.stringAlsAutomat("""
        Z: z1 z2
        E: z2
        S: z1
        A: a b 
        z1 a z2
        z1 b z2
        z2 a z2
        z2 b z2
        """);
    EndlicherAutomat ea2 = new EndlicherAutomat();
    ea2.stringAlsAutomat("""
        Z: z11 z12
        E: z12
        S: z11
        A: a b 
        z11 a z11
        z11 b z12
        z12 a z12
        z12 b z12
        """);
    
    Assertions.assertFalse(ea.istMinimalIsomorph(ea2));
    Assertions.assertTrue(ea.istMinimalIsomorph(ea));
    Assertions.assertFalse(ea2.istMinimalIsomorph(ea));
    Assertions.assertTrue(ea2.istMinimalIsomorph(ea2));
  }
  
  @Test
  public void testIstMinimalIsomorph4negativ() {
    EndlicherAutomat ea = new EndlicherAutomat();
    ea.stringAlsAutomat("""
        Z: z1 z2
        E: z2
        S: z1
        A: a b 
        z1 a z2
        z1 b z2
        z2 a z2
        z2 b z2
        """);
    EndlicherAutomat ea2 = new EndlicherAutomat();
    ea2.stringAlsAutomat("""
        Z: z11 z12
        E: z11 z12
        S: z11 
        A: a b 
        z11 a z12
        z11 b z12
        z12 a z12
        z12 b z12
        """);
    
    Assertions.assertFalse(ea.istMinimalIsomorph(ea2));
    Assertions.assertTrue(ea.istMinimalIsomorph(ea));
    Assertions.assertFalse(ea2.istMinimalIsomorph(ea));
    Assertions.assertTrue(ea2.istMinimalIsomorph(ea2));
  }
  
  @Test
  public void testIstMinimalIsomorph5negativ() {
    EndlicherAutomat ea = new EndlicherAutomat();
    ea.stringAlsAutomat("""
        Z: z1 z2
        E: z1
        S: z1
        A: a b 
        z1 a z2
        z1 b z2
        z2 a z2
        z2 b z2
        """);
    EndlicherAutomat ea2 = new EndlicherAutomat();
    ea2.stringAlsAutomat("""
        Z: z11 z12
        E: z12 z11
        S: z11 
        A: a b 
        z11 a z12
        z11 b z12
        z12 a z12
        z12 b z12
        """);
    
    Assertions.assertFalse(ea.istMinimalIsomorph(ea2));
    Assertions.assertTrue(ea.istMinimalIsomorph(ea));
    Assertions.assertFalse(ea2.istMinimalIsomorph(ea));
    Assertions.assertTrue(ea2.istMinimalIsomorph(ea2));
  }
  
  @Test
  public void testIstMinimalIsomorph6negativ() {
    EndlicherAutomat ea = RegulaererAusdruck.alsAusdruck("(aaab)*", true).alsAutomat().deterministisch().minimieren().zustandsnamenKuerzen();  
    EndlicherAutomat ea2 = RegulaererAusdruck.alsAusdruck("(aaaab)*", true).alsAutomat().deterministisch().minimieren().zustandsnamenKuerzen();  
    Assertions.assertFalse(ea.istMinimalIsomorph(ea2));
    Assertions.assertFalse(ea2.istMinimalIsomorph(ea));
    Wort w1 = ea.nichtGemeinsamesWort(ea2);
    Assertions.assertFalse(ea.akzeptieren(w1) && ea2.akzeptieren(w1));
    w1 = ea2.nichtGemeinsamesWort(ea);
    Assertions.assertFalse(ea.akzeptieren(w1) && ea2.akzeptieren(w1));
  }
  
  @Test
  public void testIstMinimalIsomorph7negativ() {
    EndlicherAutomat ea = new EndlicherAutomat();
    ea.stringAlsAutomat("""
        A: a b
    Z: z0 z1 z2 z3 z1_z3 z0_z3 leer
    S: z0
    E: z3 z0_z3 z1_z3
    z0 a z3
    z0 b z1
    z1 a z0_z3
    z1 b z2
    z2 a z1_z3
    z2 b z3
    z3 a leer
    z3 b leer
    leer a leer
    leer b leer
    z0_z3 a z3
    z0_z3 b z1
    z1_z3 a z3
    z1_z3 b z2
        """);
  ea = ea.minimieren();
  EndlicherAutomat ea2 = new EndlicherAutomat();
    ea2.stringAlsAutomat("""
        A: a b
        Z: z0 z1 z2 z3
        E: z3
        S: z0
        z0 a z3
        z0 b z1
        z1 a z0
        z1 a z3
        z1 b z2
        z2 a z1
        z2 a z3
        z2 b z3
        """);
    ea2 = ea2.deterministisch(false).minimieren();
    Assertions.assertFalse(ea.istMinimalIsomorph(ea2));
    Assertions.assertFalse(ea2.istMinimalIsomorph(ea));
    Wort w1 = ea.nichtGemeinsamesWort(ea2);
    Assertions.assertFalse(ea.akzeptieren(w1) && ea2.akzeptieren(w1));
    w1 = ea2.nichtGemeinsamesWort(ea);
    Assertions.assertFalse(ea.akzeptieren(w1) && ea2.akzeptieren(w1));
  }
  
  @Test
  public void testEpsilonEntfernenBeispiel1() {
    EndlicherAutomat ea = new EndlicherAutomat();
    ea.stringAlsAutomat("""
        Z: z0 z1 z2
        E: z1
        S: z0
        A: a b
        z0 /eps z1
        z0 /eps z2
        z0 a z2
        z0 a z1
        z0 b z2
        z2 /eps z2
        z2 b z2
        z2 a z2
        z1 a z1
        """);
    Assertions.assertTrue(ea.akzeptieren("a"));
    Assertions.assertTrue(ea.akzeptieren("aa"));
    Assertions.assertFalse(ea.akzeptieren("ab"));
    
    EndlicherAutomat ea2 = ea.epsilonEntfernen();
    Assertions.assertTrue(ea2.akzeptieren("a"));
    Assertions.assertTrue(ea2.akzeptieren("aa"));
    Assertions.assertFalse(ea2.akzeptieren("ab"));
    
    ea2 = ea2.deterministisch().minimieren();
    
    EndlicherAutomat ea3 = RegulaererAusdruck.alsAusdruck("(a)*", true)
        .alsAutomat();
    ea3.addZeichen("b");
    ea3 = ea3.deterministisch().minimieren();
    Assertions.assertTrue(ea2.istMinimalIsomorph(ea3), ea2.zustandsnamenKuerzen() + "\n\n" + ea3.zustandsnamenKuerzen());
    Assertions.assertTrue(ea3.istMinimalIsomorph(ea2));
  }
  
  @Test
  public void testEpsilonEntfernenBeispiel2() {
    EndlicherAutomat ea = new EndlicherAutomat();
    ea.stringAlsAutomat("""
        Z: z0 z1 z2 z3 z4 z5 z6 z7
        E: z7
        S: z0
        A: a b c 
        z0 /eps z1
        z0 /eps z2
        z0 a z3
        z1 /eps z4
        z2 /eps z4
        z2 b z5
        z3 b z5
        z4 /eps z6
        z4 c z7
        z5 /eps z7
        """);
    Assertions.assertTrue(ea.akzeptieren("c"));
    Assertions.assertTrue(ea.akzeptieren("b"));
    Assertions.assertTrue(ea.akzeptieren("ab"));
    
    EndlicherAutomat ea2 = ea.epsilonEntfernen();
    Assertions.assertTrue(ea2.akzeptieren("c"));
    Assertions.assertTrue(ea2.akzeptieren("b"));
    Assertions.assertTrue(ea2.akzeptieren("ab"));
    
    ea2 = ea2.deterministisch().minimieren();
    
    EndlicherAutomat ea3 = RegulaererAusdruck.alsAusdruck("((c + b) + ab)", true)
        .alsAutomat().deterministisch().minimieren();
    Assertions.assertTrue(ea2.istMinimalIsomorph(ea3));
    Assertions.assertTrue(ea3.istMinimalIsomorph(ea2));
  }
  
  @Test
  public void testUnterschiedlicheWorteTeilmenge() {
    EndlicherAutomat ea = new EndlicherAutomat();
    ea.stringAlsAutomat("""
        Z: z1 z2
        E: z1
        S: z1
        A: a b 
        z1 a z2
        z1 b z2
        z2 a z2
        z2 b z2
        """);
    ea.deterministisch().minimieren().zustandsnamenKuerzen();  
    EndlicherAutomat ea2 = new EndlicherAutomat();
    ea2.stringAlsAutomat("""
        Z: z1 z2
        E: z1 z2
        S: z1
        A: a b 
        z1 a z2
        z1 b z2
        z2 a z2
        z2 b z2
        """);
    ea2.deterministisch().minimieren().zustandsnamenKuerzen(); 
    Wort w1 = ea.nichtGemeinsamesWort(ea2);
    Assertions.assertFalse(ea.akzeptieren(w1) && ea2.akzeptieren(w1));
    w1 = ea2.nichtGemeinsamesWort(ea);
    Assertions.assertFalse(ea.akzeptieren(w1) && ea2.akzeptieren(w1));
  }
  
  public void testUnterschiedlicheWorte() {
    EndlicherAutomat ea = new EndlicherAutomat();
    ea.stringAlsAutomat("""
        Z: z1 z2
        E: z2
        S: z1
        A: a b 
        z1 a z2
        z1 b z2
        z2 a z2
        z2 b z2
        """);
    ea.deterministisch().minimieren().zustandsnamenKuerzen();  
    EndlicherAutomat ea2 = new EndlicherAutomat();
    ea2.stringAlsAutomat("""
        Z: z1 z2
        E: 
        S: z1
        A: a b 
        z1 a z2
        z1 b z2
        z2 a z2
        z2 b z2
        """);
    ea2.deterministisch().minimieren().zustandsnamenKuerzen(); 
    Wort w1 = ea.nichtGemeinsamesWort(ea2);
    Assertions.assertFalse(ea.akzeptieren(w1) && ea2.akzeptieren(w1));
    w1 = ea2.nichtGemeinsamesWort(ea);
    Assertions.assertFalse(ea.akzeptieren(w1) && ea2.akzeptieren(w1));
  }
  
  public void testUnterschiedlicheWorteException() {
    EndlicherAutomat ea = new EndlicherAutomat();
    ea.stringAlsAutomat("""
        Z: z1 z2
        E: z2
        S: z1
        A: a b 
        z1 a z2
        z1 b z2
        z2 a z2
        z2 b z2
        """);
    //ea.deterministisch().minimieren().zustandsnamenKuerzen();  
    EndlicherAutomat ea2 = new EndlicherAutomat();
    ea2.stringAlsAutomat("""
        Z: z1 z2 z3
        E: z2 z3
        S: z1
        A: a b 
        z1 a z2
        z1 b z2
        z2 a z3
        z2 b z2
        z3 a z2
        z3 b z3
        """);
    //ea2.deterministisch().minimieren().zustandsnamenKuerzen(); 
    try {
      Wort w = ea.nichtGemeinsamesWort(ea2);
      Assertions.fail(w + " ist gemeinsames Wort der Automaten");
    } catch(IllegalStateException e) {}
    try {
      Wort w = ea2.nichtGemeinsamesWort(ea);
      Assertions.fail(w + " ist gemeinsames Wort der Automaten");
    } catch(IllegalStateException e) {}
  }
  
  public void testUnterschiedlicheWorteExceptionRegulaer() {
    EndlicherAutomat ea = RegulaererAusdruck.alsAusdruck("(a)*(b)*)*a", true).alsAutomat().deterministisch();
    EndlicherAutomat ea2 = RegulaererAusdruck.alsAusdruck("(a+b)*a", true).alsAutomat().deterministisch();
    try {
      Wort w = ea.nichtGemeinsamesWort(ea2);
      Assertions.fail(w + " ist gemeinsames Wort der Automaten");
    } catch(IllegalStateException e) {}
    try {
      Wort w = ea2.nichtGemeinsamesWort(ea);
      Assertions.fail(w + " ist gemeinsames Wort der Automaten");
    } catch(IllegalStateException e) {}
  }
  
  @Test
  public void testUnterschiedlicheWorteRegulaer1() {
    EndlicherAutomat ea = RegulaererAusdruck.alsAusdruck("(abab)*", true).alsAutomat().deterministisch(); //.minimieren().zustandsnamenKuerzen();  
    EndlicherAutomat ea2 = RegulaererAusdruck.alsAusdruck("(ab)*", true).alsAutomat().deterministisch(); //.minimieren().zustandsnamenKuerzen();  
    Assertions.assertFalse(ea.istMinimalIsomorph(ea2));
    Assertions.assertFalse(ea2.istMinimalIsomorph(ea));
    Wort w1 = ea.nichtGemeinsamesWort(ea2);
    Assertions.assertFalse(ea.akzeptieren(w1) && ea2.akzeptieren(w1));
    w1 = ea2.nichtGemeinsamesWort(ea);
    Assertions.assertFalse(ea.akzeptieren(w1) && ea2.akzeptieren(w1));
  }
}
