package test.kontextfreieGrammatik;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.Assertions;

import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

import alphabet.Nichtterminal;
import alphabet.Terminal;
import alphabet.Wort;
import alphabet.Zeichen;
import ausfuehrung.Ableitung;
import ausfuehrung.CYKKnoten;
import grammatik.KontextfreieGrammatik;
import grammatik.KontextfreieRegel;
import zustand.Zustand;

public class KontextfreieGrammatikTest {

  @BeforeEach
  public void setUp() throws Exception {
    Zustand.reset();
    Terminal.reset();
    Nichtterminal.reset();
    Zeichen.reset();
  }
  
  @AfterEach
  public void tearDown() {
    Terminal.reset();
    Nichtterminal.reset();
  }
  
  @Test
  public void testTerminaleUndNichtterminale() {
    Terminal t1 = Terminal.terminal("a");
    Nichtterminal n1 = Nichtterminal.nichtterminal("A");
    Assertions.assertTrue(Zeichen.istZeichen("a"));
    Assertions.assertTrue(Zeichen.istZeichen("A"));
    Assertions.assertTrue(Terminal.istTerminal("a"));
    Assertions.assertFalse(Terminal.istTerminal("A"));
    Assertions.assertTrue(Nichtterminal.istNichtterminal("A"));
    Assertions.assertFalse(Nichtterminal.istNichtterminal("a"));
    Assertions.assertSame(t1, Zeichen.zeichen(t1.getZeichen()));
    Assertions.assertSame(t1, Terminal.terminal(t1.getZeichen()));
    Assertions.assertSame(n1, Zeichen.zeichen(n1.getZeichen()));
    Assertions.assertSame(n1, Nichtterminal.nichtterminal(n1.getZeichen()));
    try {
      Terminal.terminal("a1");
      Assertions.fail();
    } catch(IllegalStateException e) {}
    try {
      Terminal.terminal("A1");
      Assertions.fail();
    } catch(IllegalStateException e) {}
    try {
      Nichtterminal.nichtterminal("a1");
      Assertions.fail();
    } catch(IllegalStateException e) {}
    try {
      Nichtterminal.nichtterminal("A1");
      Assertions.fail();
    } catch(IllegalStateException e) {}
    Nichtterminal.reset();
    Assertions.assertTrue(Zeichen.istZeichen("a"));
    Assertions.assertTrue(Terminal.istTerminal("a"));
    Assertions.assertFalse(Nichtterminal.istNichtterminal("A"));
    Terminal.reset();
    Assertions.assertFalse(Zeichen.istZeichen("a"));
    Assertions.assertFalse(Terminal.istTerminal("a"));
    
  }
  
  @Test
  public void testElementareFunktionalitaet() {
    String grammatik =
        """
        N: AX BX
        T: a b 
        S: AX
        r1:: AX -> BX | a
        r2:: BX -> b 
        r3:: BX -> /eps
        """;
    KontextfreieGrammatik g = new KontextfreieGrammatik();
    g.stringAlsGrammatik(grammatik);
    Set<Nichtterminal> nt = g.getNichtterminale();
    Assertions.assertEquals(2, nt.size());
    Assertions.assertTrue(nt.contains(Nichtterminal.nichtterminal("AX")));
    Assertions.assertTrue(nt.contains(Nichtterminal.nichtterminal("BX")));
    Set<Terminal> t = g.getTerminale();
    Assertions.assertEquals(2, t.size());
    Assertions.assertTrue(t.contains(Terminal.terminal("a")));
    Assertions.assertTrue(t.contains(Terminal.terminal("b")));
    List<KontextfreieRegel> r = g.getRegeln();
    Assertions.assertEquals(4, r.size());
    KontextfreieRegel r10 = g.regelZuId("r1_0");
    KontextfreieRegel r11 = g.regelZuId("r1_1");
    Assertions.assertEquals(Nichtterminal.nichtterminal("AX"), r10.getLinks());
    Assertions.assertEquals(Nichtterminal.nichtterminal("AX"), r11.getLinks());
    Assertions.assertEquals(g.stringAlsWort("BX"), r10.getRechts());
    Assertions.assertEquals(g.stringAlsWort("a"), r11.getRechts());
    Assertions.assertEquals(g.stringAlsWort(""), g.regelZuId("r3").getRechts());
  }

  @Test
  public void testNachLeeremWortAbleitbarCYK() {
//    S -> A
//    A -> B
//    B -> S | /eps | aa
    KontextfreieGrammatik g = new KontextfreieGrammatik();
    g.dateiEinlesen("beispiele" + util.Util.FS + "kontextfreiegrammatiken" + util.Util.FS + "testNachLeeremWortAbleitbar.kfg");
    //System.out.println(g);
    Map<Nichtterminal, CYKKnoten> erg = g.nachLeeremWortAbleitbarCYK();
    var a = Nichtterminal.nichtterminal("A");
    var b = Nichtterminal.nichtterminal("B");
    var s = Nichtterminal.nichtterminal("S");
    Assertions.assertEquals(1, erg.get(b).getRegeln().size());
    Assertions.assertEquals(2, erg.get(a).getRegeln().size());
    Assertions.assertEquals(3, erg.get(s).getRegeln().size());    
  }
  
  @Test
  public void testOhneLeeresWortRegelnCYK() {
//    N: S A B
//    T: a b
//    S: S
//    S -> A | ABb | bBA
//    A -> B
//    B -> S | /eps | aa
    KontextfreieGrammatik g = new KontextfreieGrammatik();
    g.dateiEinlesen("beispiele" + util.Util.FS + "kontextfreiegrammatiken" + util.Util.FS + "testohneLeeresWortRegelnCYK.kfg");
//    System.out.println(g);
    boolean eps = g.ohneLeeresWortRegelnCYK();
//    Map<KontextfreieRegel, List<Regelnutzung>> erg = g.getErgaenzteRegeln();
//    System.out.println(erg);
//    var an = Nichtterminal.getNichtterminal("A");
//    var bn = Nichtterminal.getNichtterminal("B");
//    var sn = Nichtterminal.getNichtterminal("S");
//    var b = Terminal.getTerminal("b");
    List<Wort> tmp = List.of(g.stringAlsWort("Ab")
        , g.stringAlsWort("Bb"), g.stringAlsWort("bB")
        , g.stringAlsWort("bA"), g.stringAlsWort("b")
        , g.stringAlsWort("ABb"), g.stringAlsWort("bBA")
        , g.stringAlsWort("B"), g.stringAlsWort("A")
        , g.stringAlsWort("S"), g.stringAlsWort("aa")
    );
    List<Wort> rechts = new ArrayList<>(); // so veraenderbar
    rechts.addAll(tmp);
    Assertions.assertTrue(eps, "Sprache enthaelt leeres Wort");   
    for(KontextfreieRegel reg: g.getRegeln()) {
      Assertions.assertNotEquals(new Wort(), reg.getRechts());
      rechts.remove(reg.getRechts());
    }
    Assertions.assertTrue(rechts.isEmpty(), rechts + " sollte bei einer "
        + "Regel auf der rechten Seite stehen");
//    System.out.println(g);
  }
  
  @Test
  public void testEntferneKettenregelnCYK() {
//    N: S A B C
//    T: a b c
//    S: S
//    S -> A
//    A -> B | C | aa
//    B -> bb
//    C -> S | /eps | cc
    KontextfreieGrammatik g = new KontextfreieGrammatik();
    g.dateiEinlesen("beispiele" + util.Util.FS + "kontextfreiegrammatiken" + util.Util.FS + "testEntferneKettenregeln.kfg");
    //System.out.println(g);
    g.entferneKettenregelnCYK();
    //System.out.println(g);
    //Map<KontextfreieRegel, List<Regelnutzung>> erg = g.getErgaenzteRegeln();
    //erg.forEach((k,v) -> System.out.println(k + " : " + v));
    //System.out.println(erg);
    var an = Nichtterminal.nichtterminal("A");
    var cn = Nichtterminal.nichtterminal("C");
    var sn = Nichtterminal.nichtterminal("S");

    List<KontextfreieRegel> tmp = List.of(
        new KontextfreieRegel(an, g.stringAlsWort("bb"))
        , new KontextfreieRegel(an, g.stringAlsWort(""))
        , new KontextfreieRegel(an, g.stringAlsWort("cc"))
        , new KontextfreieRegel(an, g.stringAlsWort("aa"))
        , new KontextfreieRegel(sn, g.stringAlsWort("aa"))
        , new KontextfreieRegel(sn, g.stringAlsWort("bb"))
        , new KontextfreieRegel(sn, g.stringAlsWort(""))
        , new KontextfreieRegel(sn, g.stringAlsWort("cc"))
        , new KontextfreieRegel(cn, g.stringAlsWort("aa"))
        , new KontextfreieRegel(cn, g.stringAlsWort("bb"))
        , new KontextfreieRegel(cn, g.stringAlsWort("cc"))
        , new KontextfreieRegel(cn, g.stringAlsWort(""))
    );
    List<KontextfreieRegel> regeln = new ArrayList<>(); // so veraenderbar
    regeln.addAll(tmp);  
    for(KontextfreieRegel reg: g.getRegeln()) {
      Assertions.assertFalse(reg.getRechts().laenge() == 1 && reg.getRechts().istNichtterminal());
      regeln.remove(reg);
    }
    Assertions.assertTrue(regeln.isEmpty(), regeln + " sollte es u. a. als "
        + "Regeln geben");
//    System.out.println(g);
  }
  
  @Test
  public void testMehrereZeichenLeeremWort() {
    String grammatik =
        """
        N: A B
        T: a b c 
        S: A
        r1:: A -> AcB
        r2:: A -> A | BB | a 
        r3:: B -> b | /eps
        """;
    String[] woerter= {"", "a", "b", "bb", "c", "ac", "cb", "acb", "bbc", "bbcb"};
    //String[] woerter= {"a", "c", "b", "bb", "ac", "cb", "acb", "bbc", "bbcb", ""};

    for (String s: woerter) {
      Terminal.reset();
      Nichtterminal.reset();
      var g = new KontextfreieGrammatik();
      g.stringAlsGrammatik(grammatik);
      //g.ohneLeeresWortRegelnCYK();
      Wort w = g.stringAlsWort(s);
      System.out.println("  Wort: " + w);
      Ableitung ab = g.berechneAbleitung(w);
      Assertions.assertTrue(ab.isErfolgreich());
      Assertions.assertEquals(w, ab.getAbgeleitet());
    }
  }
  
  @Test
  public void testInChomskyNormalformCYK() {
//    N: S A B C D
//    T: a b c
//    S: S
//    S -> ABCD | AB | abc | aaBBbb
    KontextfreieGrammatik g = new KontextfreieGrammatik();
    g.dateiEinlesen("beispiele" + util.Util.FS + "kontextfreiegrammatiken" + util.Util.FS + "testInChomskyNormalformCYK.kfg");
    //System.out.println(g);
    g = g.inChomskyNormalformCYK();
    //System.out.println(g);
    List<Wort> tmp = List.of(g.stringAlsWort("CD")
        , g.stringAlsWort("AB"), g.stringAlsWort("a")
        , g.stringAlsWort("b"), g.stringAlsWort("c")
    );
    List<Wort> rechts = new ArrayList<>(); // so veraenderbar
    rechts.addAll(tmp); 
    for(KontextfreieRegel reg: g.getRegeln()) {
      int lang = reg.getRechts().laenge();
      if (lang == 1 && !reg.getRechts().terminal()) {
        Assertions.fail(reg + " darf rechts nur ein Terminal haben");
      }
      if (lang == 2 && ((! (reg.getRechts().at(0) instanceof Nichtterminal)
          || ! (reg.getRechts().at(1) instanceof Nichtterminal)))){
        Assertions.fail(reg + " darf rechts nur zwei Nichtterminale haben");
      }
      if(lang == 0 && !reg.getLinks().equals(g.getStart())){
        Assertions.fail("Leeres Wort darf nur vom Startsymbol abgeleitet werden");
      }
      if(lang > 2){
        Assertions.fail(reg  + "darf rechts maximal 2 Zeichen haben");
      }
      rechts.remove(reg.getRechts());
    }
    Assertions.assertTrue(g.istInChomskyNormalform());
    Assertions.assertTrue(rechts.isEmpty(), rechts + " sollte es in "
        + "Regeln geben");
//    System.out.println(g);
  }

  @Test
  public void testInChomskyNormalformCYK2() {
    KontextfreieGrammatik g = new KontextfreieGrammatik();
    g.dateiEinlesen("beispiele" + util.Util.FS + "kontextfreiegrammatiken" + util.Util.FS + "prog.kfg");
    //System.out.println(g);
    g = g.inChomskyNormalformCYK();
    for(KontextfreieRegel reg: g.getRegeln()) {
      int lang = reg.getRechts().laenge();
      if (lang == 1 && !reg.getRechts().terminal()) {
        Assertions.fail(reg + " darf rechts nur ein Terminal haben");
      }
      if (lang == 2 && ((! (reg.getRechts().at(0) instanceof Nichtterminal)
          || ! (reg.getRechts().at(1) instanceof Nichtterminal)))){
        Assertions.fail(reg + " darf rechts nur zwei Nichtterminale haben");
      }
      if(lang == 0 && !reg.getLinks().equals(g.getStart())){
        Assertions.fail("Leeres Wort darf nur vom Startsymbol abgeleitet werden");
      }
      if(lang > 2){
        Assertions.fail(reg  + "darf rechts maximal 2 Zeichen haben");
      }
    }
    Assertions.assertTrue(g.istInChomskyNormalform());
  }
}
