package test.turingMaschine;

import java.io.File;
import java.util.Date;

import org.junit.jupiter.api.Assertions;

import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

import alphabet.Nichtterminal;
import alphabet.Terminal;
import alphabet.Wort;
import alphabet.Zeichen;
import turingMaschine.Konfiguration;
import turingMaschine.Richtung;
import turingMaschine.TMModusBand;
import turingMaschine.TMModusStart;
import turingMaschine.TuringMaschine;
import zustand.Zustand;

public class TuringMaschineTest {
  
  @BeforeEach
  public void setUp() throws Exception {
    Zustand.reset();
    Terminal.reset();
    Nichtterminal.reset();
    Zeichen.reset();
  }

  
  @Test
  public void testEinfacheMaschine() {
    TuringMaschine tm = new TuringMaschine();
    tm.addZeichen("a", "b");
    tm.addZustand("Start", "z1", "S");
    Zustand start = Zustand.zustand("Start");
//    Zustand z1 = Zustand.zustand("z1");
//    Zustand s = Zustand.zustand("S");
    tm.setStart(start);
    tm.addUeberfuehrung("Start", "#", "z1", "#", Richtung.LINKS);
    tm.addUeberfuehrung("z1", "a", "z1", "b", Richtung.LINKS);
    tm.addUeberfuehrung("z1", "b", "z1", "a", Richtung.LINKS);
    tm.addUeberfuehrung("z1", "#", "S", "#", Richtung.STOPP);
    tm.bearbeite("aba");
    Wort erg = tm.ausfuehren();
    Assertions.assertEquals(new Wort(""),  erg);
    Konfiguration kf = tm.getKonfiguration();
    Assertions.assertEquals(Zustand.zustand("S"), kf.getZustand());
    Assertions.assertEquals(Zeichen.zeichen("#"), kf.getKopf());
    Assertions.assertEquals(new Wort(""), kf.getLinks());
    Assertions.assertTrue(kf.getRechts().toString().startsWith("bab"));
    Assertions.assertTrue(tm.isHalt());
    Assertions.assertFalse(tm.isDivergiert());
    Assertions.assertFalse(tm.isHaengt());
  }
  
  @Test
  public void testMaschineLaeuftAnDieWand() {
    TuringMaschine tm = new TuringMaschine();
    tm.addZeichen("a", "b");
    tm.addZustand("Start", "z1", "S");
    Zustand start = Zustand.zustand("Start");
    tm.setStart(start);
    tm.addUeberfuehrung("Start", "#", "z1", "#", Richtung.LINKS);
    tm.addUeberfuehrung("z1", "a", "z1", "b", Richtung.LINKS);
    tm.addUeberfuehrung("z1", "b", "z1", "a", Richtung.LINKS);
    tm.addUeberfuehrung("z1", "#", "z1", "#", Richtung.LINKS);
    tm.bearbeite("aba");
    Wort erg = tm.ausfuehren();
    Assertions.assertEquals(new Wort("#"),  erg);
    Konfiguration kf = tm.getKonfiguration();
    Assertions.assertEquals(Zustand.zustand("z1"), kf.getZustand());
    Assertions.assertEquals(Zeichen.zeichen("b"), kf.getKopf());
    Assertions.assertEquals(new Wort("#"), kf.getLinks());
    Assertions.assertTrue(kf.getRechts().toString().startsWith("ab")); 
    Assertions.assertTrue(tm.isHaengt());
    Assertions.assertFalse(tm.isDivergiert());
    Assertions.assertFalse(tm.isHalt());
  }
  
  @Test
  public void testMaschineLaeuftAnDieWandAberBandUnendlich() {
    TuringMaschine tm = new TuringMaschine(TMModusStart.HINTERDEMENDE
        , TMModusBand.UNBESCHRAENKT);
    tm.addZeichen("a", "b");
    tm.addZustand("Start", "z1", "z2", "z3", "S");
    Zustand start = Zustand.zustand("Start");
    tm.setStart(start);
    tm.addUeberfuehrung("Start", "#", "z1", "#", Richtung.LINKS);
    tm.addUeberfuehrung("z1", "a", "z1", "b", Richtung.LINKS);
    tm.addUeberfuehrung("z1", "b", "z1", "a", Richtung.LINKS);
    tm.addUeberfuehrung("z1", "#", "z2", "#", Richtung.LINKS);
    tm.addUeberfuehrung("z2", "#", "z3", "#", Richtung.LINKS);
    tm.addUeberfuehrung("z3", "#", "z2", "#", Richtung.RECHTS);
    tm.bearbeite("aba");
    //System.out.println("Ergebnis: " + tm.berechnen());
    Wort erg = tm.ausfuehren();
    Assertions.assertEquals(new Wort(""),  erg);
    Konfiguration kf = tm.getKonfiguration();
    Assertions.assertEquals(Zustand.zustand("z3"), kf.getZustand());
    Assertions.assertEquals(Zeichen.zeichen("#"), kf.getKopf());
    Assertions.assertEquals(new Wort(""), kf.getLinks());
    Assertions.assertTrue(kf.getRechts().toString().contains("bab")); 
    Assertions.assertFalse(tm.isHaengt());
    Assertions.assertTrue(tm.isDivergiert());
    Assertions.assertFalse(tm.isHalt());
  }
  
  @Test
  public void testMaschineDivergiert() {
    TuringMaschine tm = new TuringMaschine();
    tm.addZeichen("a", "b");
    tm.addZustand("Start", "z1", "z2", "S");
    Zustand start = Zustand.zustand("Start");
    tm.setStart(start);
    tm.addUeberfuehrung("Start", "#", "z1", "#", Richtung.LINKS);
    tm.addUeberfuehrung("z1", "a", "z2", "a", Richtung.LINKS);
    tm.addUeberfuehrung("z1", "b", "z2", "b", Richtung.LINKS);
    tm.addUeberfuehrung("z2", "a", "z1", "a", Richtung.RECHTS);
    tm.addUeberfuehrung("z2", "b", "z1", "b", Richtung.RECHTS);
    tm.bearbeite("aba");
    //System.out.println("Ergebnis: " + tm.berechnen());
    tm.ausfuehren();
    Assertions.assertTrue(tm.isDivergiert());
    Assertions.assertFalse(tm.isHaengt());
    Assertions.assertFalse(tm.isHalt());
  }
  
  @Test
  public void testTuringMaschineAusString() {
    TuringMaschine tm = new TuringMaschine();
    System.out.println("Zeichen: " + Zeichen.istZeichen("#"));
    tm.stringAlsTuringMaschine("""
        % Zustaende
        Z: Start z1 S
        % Alphabet, Leerzeichen # automatisch dabei
        A: a b
        % Start
        S: Start
        % Ueberfuehrungsfunktion
        Start #   z1 # L
        z1    a   z1 b L
        z1    b   z1 a L
        z1    #   S  # S
        """);
    tm.bearbeite("aba");
    //System.out.println("Ergebnis: " + tm.berechnen());
    Wort erg = tm.ausfuehren();
    Assertions.assertEquals(new Wort(""),  erg);
    Konfiguration kf = tm.getKonfiguration();
    Assertions.assertEquals(Zustand.zustand("S"), kf.getZustand());
    Assertions.assertEquals(Zeichen.zeichen("#"), kf.getKopf());
    Assertions.assertEquals(new Wort(""), kf.getLinks());
    Assertions.assertTrue(kf.getRechts().toString().startsWith("bab"));
    Assertions.assertTrue(tm.isHalt());
    Assertions.assertFalse(tm.isDivergiert());
    Assertions.assertFalse(tm.isHaengt());
  }
  
  @Test
  public void testTuringMaschineAusDatei() {
    TuringMaschine tm = new TuringMaschine();
    //System.out.println("Zeichen: " + Zeichen.istZeichen("#"));
    tm.dateiEinlesen("beispiele" + util.Util.FS + "turingmaschinen" + util.Util.FS + "Erste.tm");
    tm.bearbeite("aba");
    //System.out.println("Ergebnis: " + tm.berechnen());
    Wort erg = tm.ausfuehren();
    Assertions.assertEquals(new Wort(""),  erg);
    Konfiguration kf = tm.getKonfiguration();
    Assertions.assertEquals(Zustand.zustand("S"), kf.getZustand());
    Assertions.assertEquals(Zeichen.zeichen("#"), kf.getKopf());
    Assertions.assertEquals(new Wort(""), kf.getLinks());
    Assertions.assertTrue(kf.getRechts().toString().startsWith("bab"));
    Assertions.assertTrue(tm.isHalt());
    Assertions.assertFalse(tm.isDivergiert());
    Assertions.assertFalse(tm.isHaengt());
  }
  
  @Test
  public void testMaschineInDateiUndAusDatei() {
    String datei = new Date().getTime() + ".tm"; 
    TuringMaschine tm0 = new TuringMaschine();
    tm0.addZeichen("a", "b");
    tm0.addZustand("Start", "z1", "S");
    Zustand start = Zustand.zustand("Start");
    tm0.setStart(start);
    tm0.addUeberfuehrung("Start", "#", "z1", "#", Richtung.LINKS);
    tm0.addUeberfuehrung("z1", "a", "z1", "b", Richtung.LINKS);
    tm0.addUeberfuehrung("z1", "b", "z1", "a", Richtung.LINKS);
    tm0.addUeberfuehrung("z1", "#", "S", "#", Richtung.STOPP);
    tm0.dateiSpeichern(datei);
    
    TuringMaschine tm = new TuringMaschine();
    tm.dateiEinlesen(datei);
    tm.bearbeite("aba");
    //System.out.println("Ergebnis: " + tm.berechnen());
    Wort erg = tm.ausfuehren();
    Assertions.assertEquals(new Wort(""),  erg);
    Konfiguration kf = tm.getKonfiguration();
    Assertions.assertEquals(Zustand.zustand("S"), kf.getZustand());
    Assertions.assertEquals(Zeichen.zeichen("#"), kf.getKopf());
    Assertions.assertEquals(new Wort(""), kf.getLinks());
    Assertions.assertTrue(kf.getRechts().toString().startsWith("bab"));
    Assertions.assertTrue(tm.isHalt());
    Assertions.assertFalse(tm.isDivergiert());
    Assertions.assertFalse(tm.isHaengt());
    
    File tmp = new File(datei);
    tmp.delete();
  }
  
  @Test
  public void testTuringMaschineStartetLinksLaeuftAnDieWand() {
    TuringMaschine tm = new TuringMaschine(TMModusStart.AMANFANG
        , TMModusBand.LINKSENDEND);
    tm.stringAlsTuringMaschine("""
        % Zustaende
        Z: Start z1 
        % Alphabet, Leerzeichen # automatisch dabei
        A: a b
        % Start
        S: Start
        % Ueberfuehrungsfunktion
        Start a   z1 # L
        z1    #   z1 # L  // sofort gegen die Wand
        """);
    tm.bearbeite("aba");
    //System.out.println("Ergebnis: " + tm.berechnen());
    Wort erg = tm.ausfuehren();
    //System.out.println("Konfig: " + tm.getKonfiguration());
    Assertions.assertEquals(new Wort("#ba"),  erg);
    Konfiguration kf = tm.getKonfiguration();
    Assertions.assertEquals(Zustand.zustand("z1"), kf.getZustand());
    Assertions.assertEquals(Zeichen.zeichen("#"), kf.getKopf());
    Assertions.assertFalse(tm.isHalt());
    Assertions.assertFalse(tm.isDivergiert());
    Assertions.assertTrue(tm.isHaengt());
  }
  
  @Test
  public void testTuringMaschineStartetLinksWandeltWortUm() {
    TuringMaschine tm = new TuringMaschine(TMModusStart.AMANFANG
        , TMModusBand.LINKSENDEND);
    tm.stringAlsTuringMaschine("""
        % Zustaende
        Z: Start z1 zurueck last S
        % Alphabet, Leerzeichen # automatisch dabei
        A: a b
        % Start
        S: Start
        % Ueberfuehrungsfunktion
        Start   a  z1 b R
        Start   b  z1 a R
        z1      a  z1 b R  
        z1      b  z1 a R
        z1      #  zurueck # L
        zurueck a  zurueck a L
        zurueck b  zurueck b L
        zurueck #  last    # R
        last    a  S       a S 
        last    b  S       b S  
        """);
    tm.bearbeite("abba");
    //System.out.println("Ergebnis: " + tm.berechnen());
    Wort erg = tm.ausfuehren();
    System.out.println("Konfig0: " + tm.getKonfiguration());
    Assertions.assertEquals(new Wort("baab"),  erg);
    Konfiguration kf = tm.getKonfiguration();
    Assertions.assertEquals(Zustand.zustand("S"), kf.getZustand());
    Assertions.assertEquals(Zeichen.zeichen("b"), kf.getKopf());
    Assertions.assertTrue(tm.isHalt());
    Assertions.assertFalse(tm.isDivergiert());
    Assertions.assertFalse(tm.isHaengt());
  }
  
  @Test
  public void testTuringMaschineAkzeptanz() {
    TuringMaschine tm = new TuringMaschine();
    tm.stringAlsTuringMaschine("""
        % akzepiert a^n b^(n+m)
        Z: Start entferneB sucheStelleVorA entferneA sucheStelleHinterB 
        Z: pruefeAufNurB S 
        A: a b 
        S: Start
        sucheStelleVorA    a sucheStelleVorA    a LINKS  
        sucheStelleVorA    b sucheStelleVorA    b LINKS  
        sucheStelleVorA    # entferneA          # RECHTS 
        pruefeAufNurB      # S                  # STOPP  
        sucheStelleHinterB a sucheStelleHinterB a RECHTS 
        sucheStelleHinterB b sucheStelleHinterB b RECHTS 
        sucheStelleHinterB # entferneB          # LINKS  
        pruefeAufNurB      b pruefeAufNurB      # RECHTS 
        Start              # entferneB          # LINKS  
        entferneB          b sucheStelleVorA    # LINKS  
        entferneA          a sucheStelleHinterB # RECHTS 
        entferneB          # S                  # STOPP  
        entferneA          b pruefeAufNurB      # RECHTS 
        entferneA          # S                  # STOPP  
        """);
    Assertions.assertTrue(tm.akzeptieren(""));
    Assertions.assertTrue(tm.akzeptieren("b"));
    Assertions.assertTrue(tm.akzeptieren("ab"));
    Assertions.assertTrue(tm.akzeptieren("aaabbbbbbbbb"));
    Assertions.assertFalse(tm.akzeptieren("a"));
    Assertions.assertFalse(tm.akzeptieren("aab"));
    Assertions.assertFalse(tm.akzeptieren("baab"));
    Assertions.assertFalse(tm.akzeptieren("abbba"));
    Assertions.assertFalse(tm.akzeptieren("bba"));
  }
  
  @Test
  public void testTuringMaschineFehlenderZustand() {
    TuringMaschine tm = new TuringMaschine();
    try {
      tm.stringAlsTuringMaschine("""
          % Zustaende
          Z: Start z1
          % Alphabet, Leerzeichen # automatisch dabei
          A: a b
          % Start
          S: Start
          % Ueberfuehrungsfunktion
          Start #   z1 # L
          z1    a   z1 b L
          z1    b   z1 a L
          z1    #   S  # S
          """);
      Assertions.fail("fehlenden Zustand nicht gefunden");
    } catch(IllegalStateException e) {}
  }
  
  @Test
  public void testTuringMaschineFehlendesZeichenSchreiben() {
    TuringMaschine tm = new TuringMaschine();
    try {
      tm.stringAlsTuringMaschine("""
          % Zustaende
          Z: Start z1 S
          % Alphabet, Leerzeichen # automatisch dabei
          A: a 
          % Start
          S: Start
          % Ueberfuehrungsfunktion
          Start #   z1 # L
          z1    a   z1 b L
          z1    b   z1 a L
          z1    #   S  # S
          """);
      Assertions.fail("fehlendes Zeichen nicht gefunden");
    } catch(IllegalStateException e) {}
  }
  
  @Test
  public void testTuringMaschineFehlendesZeichenLesen() {
    TuringMaschine tm = new TuringMaschine();
    try {
      tm.stringAlsTuringMaschine("""
          % Zustaende
          Z: Start z1 S
          % Alphabet, Leerzeichen # automatisch dabei
          A: b 
          % Start
          S: Start
          % Ueberfuehrungsfunktion
          Start #   z1 # L
          z1    a   z1 b L
          z1    b   z1 a L
          z1    #   S  # S
          """);
      Assertions.fail("fehlendes Zeichen nicht gefunden");
    } catch(IllegalStateException e) {}
  }
  
  @Test
  public void testTuringMaschineFalscheUeberfuehrung1() {
    TuringMaschine tm = new TuringMaschine();
    try {
      tm.stringAlsTuringMaschine("""
          % Zustaende
          Z: Start z1 S
          % Alphabet, Leerzeichen # automatisch dabei
          A: a b
          % Start
          S: Start
          % Ueberfuehrungsfunktion
          Start #   z1 # L
          z1    a   z1 b X
          z1    b   z1 a L
          z1    #   S  # S
          """);
      Assertions.fail("Fehler in Ueberfuehrungsfunktion nicht gefunden");
    } catch(IllegalStateException e) {}
  }
  
  @Test
  public void testTuringMaschineFalscheUeberfuehrung2() {
    TuringMaschine tm = new TuringMaschine();
    try {
      tm.stringAlsTuringMaschine("""
          % Zustaende
          Z: Start z1 S
          % Alphabet, Leerzeichen # automatisch dabei
          A: a b
          % Start
          S: Start
          % Ueberfuehrungsfunktion
          Start #   z1 # L
          z1    a   z1 b L blubb
          z1    b   z1 a L
          z1    #   S  # S
          """);
      Assertions.fail("Fehler in Ueberfuehrungsfunktion nicht gefunden");
    } catch(IllegalStateException e) {}
  }
  
  
  @Test
  public void testTuringMaschineFalscherStartzustand() {
    TuringMaschine tm = new TuringMaschine();
    try {
      tm.stringAlsTuringMaschine("""
          % Zustaende
          Z: Start z1 S
          % Alphabet, Leerzeichen # automatisch dabei
          A: a b
          % Start
          S: start
          % Ueberfuehrungsfunktion
          Start #   z1 # L
          z1    a   z1 b L
          z1    b   z1 a L
          z1    #   S  # S
          """);
      Assertions.fail("Fehler im Startzustand nicht gefunden");
    } catch(IllegalStateException e) {}
  }
  
  
  @Test
  public void testTuringMaschineFalscherStartzustand2() {
    TuringMaschine tm = new TuringMaschine();
    try {
      tm.stringAlsTuringMaschine("""
          % Zustaende
          Z: Start z1 S
          % Alphabet, Leerzeichen # automatisch dabei
          A: a b
          % Start
          S: Start z1
          % Ueberfuehrungsfunktion
          Start #   z1 # L
          z1    a   z1 b L
          z1    b   z1 a L
          z1    #   S  # S
          """);
      Assertions.fail("Fehler im Startzustand nicht gefunden");
    } catch(IllegalStateException e) {}
  }

}
