package turingMaschine;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import alphabet.Wort;
import alphabet.Zeichen;
import util.Eingabe;
import zustand.Zustand;

public class TuringMaschine {
  private List<Zustand> zustaende = new ArrayList<>();
  private List<Zeichen> alphabet = new ArrayList<>();
  private Zustand start;
  private Ueberfuehrungsfunktion ueber;
  private Konfiguration konfiguration;
  private boolean halt;
  private boolean haengt;
  private boolean divergiert;
  private TMModusStart modusRichtung;
  private TMModusBand modusBand;
  
  private List<Konfiguration> bisher = new ArrayList<>();
  
  private static final String leerzeichen = "#";
  public static Zeichen LEERZEICHEN;
  
  /** Konstruiert eine zunächst leere Turing-Maschine, 
   * die den Randbedingungen der Veranstaltung entspricht, der
   * Schreib-Lese-Kopf steht hinter dem eingegebenen Wort,
   * das Band ist linksbeschraenkt.
   */
  public TuringMaschine() {
    this(TMModusStart.HINTERDEMENDE, TMModusBand.LINKSENDEND);
  }
  
  /** Konstruiert eine Turing-Maschine bei der unterschieden wird, ob
   * die Abarbeitung hinter dem Ende oder auf dem ersten Zeichen beginnt
   * und ob das Band nach links beschraenkt ist oder nicht. 
   * @param modusr Start hinter dem Ende oder auf erstem Zeichen der
   *               Eingabe
   * @param modusb entweder endet das Band links, wobei vor der Eingabe 
   *               genau ein Leerzeichen steht und davon links der Rand
   *               ist, oder das Band ist unbeschraenkt
   */
  public TuringMaschine(TMModusStart modusr, TMModusBand modusb) {
    this.modusRichtung = modusr;
    this.modusBand = modusb;
    if (! Zeichen.istZeichen(leerzeichen)) {
      this.alphabet.add(Zeichen.zeichen(leerzeichen));
    } else {
      this.alphabet.add(Zeichen.zeichen(leerzeichen));
    }
    LEERZEICHEN = Zeichen.zeichen(leerzeichen);
    this.ueber = new Ueberfuehrungsfunktion();
  }

  /** Liest aus der uebergebenen Datei eine Turing-Maschine ein
   * und ergaenzt alle Eintrage zur bisher existierenden Turing-
   * Maschine, der Aufruf erfolgt typischerweise nachdem ein
   * TuringMaschine-Objekt erzeugt wurde. Dollte die Datei nicht gefunden
   * werden oder nicht lesbar sein, wird eine Warnung ausgegeben.
   * @param datei Pfad zur einzulesenden Datei
   */
  public void dateiEinlesen(String datei) {
    File file = new File(datei);
    if (file.exists() && file.canRead()) {
      Path path = Paths.get(datei);
      try (BufferedReader reader = Files
          .newBufferedReader(path, Charset.forName("UTF-8"))) {
        String currentLine = null;
        StringBuilder sb = new StringBuilder(); 
        while ((currentLine = reader.readLine()) != null) {
          sb.append(currentLine + "\n");         
        }
        // System.out.println(sb);
        this.stringAlsTuringMaschine(sb.toString());
      } catch(Exception e) {
        System.err.println("Problem beim Lesen der Datei " + datei + ": " + e);
        e.printStackTrace();
      }
    } else {
      System.err.println("WARNING: " + datei 
          + " existiert: " + file.exists()
          + " lesbar: " + file.canRead()
      );
    }
  }
  
  /** Methode wird intern genutzt, um unterschiedliche Datenformate
   * auf Basis der vorhandenen Daten zu generieren. Es werden neben 
   * der Datei u. a. die Nummern der Spalten uebergeben, die die
   * Reihenfolge in der Ausgabe festlegt. Dazu muessen die Zahlen
   * 0 bis 4 vergeben werden. Ist das nicht der Fall, wird eine
   * IllegalStateException geworfen.
   * @param datei Pfad der Datei in der gespeichert werden soll
   * @param alt Nummer der Spalte des ausgehenden Zustands
   * @param lesen Nummer der Spalte des eingelesenen Zeichens
   * @param neu Nummer der Spalte des neu erreichten Zustands
   * @param schreiben Nummer der Spalte des geschriebenen Zeichens
   * @param richtung Nummer der Spalte der Bewegung der TM
   * @param trenn1 Trennsymbol nach der ersten Ausgabe
   * @param trenn2 Trennsymbol nach der zweiten Ausgabe
   * @param trenn3 Trennsymbol nach der dritten Ausgabe
   * @param trenn4 Trennsymbol nach der vierten Ausgabe
   * @param trenn5 Trennsymbol nach der fuenften Ausgabe, z. B. Zeilentrenner
   * @param leerzeichen zu verwendendes Leerzeichen
   * @param lks String mit dem TM nach links laufen soll
   * @param rts String mit dem TM nach rechts lauifen soll
   * @param stopp String mit dem die TM stehenbleibt
   * @param nachHinten String der am Ende der Datei angefuegt wird
   * @param amAnfang String der am Anfang der Ausgabe geschrieben wird 
   * @throws IllegalStateException
   */
  public void dateiSpeichern(String datei, int alt, int lesen, int neu
      , int schreiben, int richtung, String trenn1, String trenn2
      , String trenn3, String trenn4, String trenn5, String leerzeichen 
      , String lks, String rts, String stopp
      , boolean nachHinten, String... amAnfang){ 
    if(alt < 0 || alt > 4 
       ||  lesen < 0 || lesen > 4 
       ||  neu < 0 || neu > 4 
       ||  schreiben < 0 || schreiben > 4 
       ||  richtung < 0 || richtung > 4 
        ) {
      throw new IllegalStateException("alle Positionen muessen zwischen "
          + "0 und 4 einschliesslich liegen");
    }  
    
    boolean[] positionen = new boolean[5];
    for(int i = 0; i < 5; i++) {
      positionen[i] = false;
    }
    positionen[alt] = true;
    positionen[lesen] = true;
    positionen[neu] = true;
    positionen[schreiben] = true;
    positionen[richtung] = true;
    
    for(int i = 0; i < 5; i++) {
      if (positionen[i] == false) {
        throw new IllegalStateException("Position " + i + "nicht vergeben"); 
      }
    }
    
    File file = new File(datei);
    try {
      if(!file.exists()) {
        file.createNewFile();
      }
    } catch (IOException e1) {
      throw new IllegalStateException("Datei " + datei 
          + " nicht erzeugbar: " + e1);
    }
    Path path = Paths.get(datei);
    try (BufferedWriter writer = Files.newBufferedWriter(path, Charset.forName("UTF-8"))) {
      StringBuilder sb = new StringBuilder("");
      
      for(String anfang: amAnfang) {
        sb.append(anfang);
      }
      
      Ueberfuehrungsfunktion basis = this.ueber.clone();
      if(amAnfang.length == 0) { // kein eigener Anfang geschrieben
        if(nachHinten) {
          sb.append("init: startX\n");
        } else {
          sb.append("initial " + this.start + "\n");
        }
        if (!Zustand.istZustand("S")) {
          System.err.println("Warnung: Keinen Endzustand S gefunden, "
              + "accept-Zustand muss von Hand eingetragen werden");
          sb.append("accept: \n");
        } else {
          sb.append("accept: S\n");
        }
      }
      writer.write(sb.toString());
      if(!Zeichen.istZeichen(leerzeichen)) {
        this.addZeichen(leerzeichen);
      }
      if(nachHinten) {
        if(!Zustand.istZustand("startX")){
            this.addZustand("startX");
        }
        if(!Zustand.istZustand("startY")){
          this.addZustand("startY");
        }
        for(Zeichen z: this.alphabet) {
          if(!z.equals(Zeichen.zeichen(leerzeichen))
              && !z.equals(LEERZEICHEN)) {
            this.ueber.add("startX", z.toString(), "startX"
                , z.toString(), Richtung.RECHTS);
          }
        }
        this.ueber.add("startX", leerzeichen, "startY", leerzeichen
            , Richtung.RECHTS);
        this.ueber.add("startY", leerzeichen, this.start.toString()
            , leerzeichen, Richtung.LINKS);
      }
      this.ueber.inDateiFormatiert(writer, alt, lesen ,neu, schreiben, richtung
          ,trenn1, trenn2, trenn3, trenn4, trenn5, leerzeichen, lks, rts, stopp);
      this.ueber = basis;
    } catch(Exception e) {
      System.err.println("Warnung: Problem beim Schreiben in die Datei " 
          + datei + ": " + e);
      e.printStackTrace();
    }   
    
  }
  
  /** Speichert die Turing-Maschine in der uebergebenen datei, so dass
   * sie mit dateiLesen() wieder gelesen werden kann. 
   * @param datei Datei mit Pfad, in der gespeichert werden soll
   */
  public void dateiSpeichern(String datei) {
    File file = new File(datei);
    try {
      if(!file.exists()) {
        file.createNewFile();
      }
    } catch (IOException e1) {
      throw new IllegalStateException("Datei " + datei 
          + " nicht erzeugbar: " + e1);
    }
    Path path = Paths.get(datei);
    try (BufferedWriter writer = Files.newBufferedWriter(path, Charset.forName("UTF-8"))) {
      StringBuilder sb = new StringBuilder("Z: ");
      for(Zustand z: this.zustaende) {
         sb.append(z.toString() + " ");
      }
      writer.write(sb.toString() +"\n");
      StringBuilder sb2 = new StringBuilder("A: ");
      for(Zeichen z: this.alphabet) {
        if(!z.equals(LEERZEICHEN)) {
          sb2.append(z.toString() + " ");
        }
      }
      writer.write(sb2.toString() + "\n");
      writer.write("S: " + this.start +"\n");
      this.ueber.inDatei(writer);
      
    } catch(Exception e) {
      System.err.println("Warnung: Problem beim Schreiben in die Datei " 
          + datei + ": " + e);
      e.printStackTrace();
    }
  }
  
  /** Versucht die TuringMaschine in einem Format zu speichern, der
   * <a href="https://turingmachinesimulator.com/">vom Sumulator</a>
   * gelesen werden kann
   * @param datei Zieldatei mit Pfad
   */
  public void dateiSpeichernUgarte(String datei) {
    this.dateiSpeichernUgarte(datei, true);
  }
  
  public void dateiSpeichernUgarte(String datei, boolean nachHinten) {
    File file = new File(datei);
    try {
      if(!file.exists()) {
        file.createNewFile();
      }
    } catch (IOException e1) {
      throw new IllegalStateException("Datei " + datei 
          + " nicht erzeugbar: " + e1);
    }
    Path path = Paths.get(datei);
    try (BufferedWriter writer = Files.newBufferedWriter(path, Charset.forName("UTF-8"))) {
      StringBuilder sb = new StringBuilder("name: " + datei + "\n");
      if(nachHinten) {
        sb.append("init: startX\n");
      } else {
        sb.append("init: " + this.start + "\n");
      }
      if (!Zustand.istZustand("S")) {
        System.err.println("Warnung: Keinen Endzustand S gefunden, "
            + "accept-Zustand muss von Hand eingetragen werden");
        sb.append("accept: \n");
      } else {
        sb.append("accept: S\n");
      }
      writer.write(sb.toString());
      if(nachHinten) {
        for(Zeichen z: this.alphabet) {
          writer.write("startX," + z +"\n");
          writer.write("startX," + z +",>\n\n");
        }
        writer.write("startX,_\n");
        writer.write(this.start + ",_,-\n\n");
      }
      this.ueber.inDateiUgarte(writer);
    } catch(Exception e) {
      System.err.println("Warnung: Problem beim Schreiben in die Datei " 
          + datei + ": " + e);
      e.printStackTrace();
    }
  }
  
  /** Versucht die TuringMaschine in einem Format zu speichern, der
   * <a href="https://github.com/schaetzc/tursi/releases">vom Sumulator</a>
   * gelesen werden kann
   * @param datei Zieldatei mit Pfad
   */
  public void dateiSpeichernTursi(String datei) {
    this.dateiSpeichernTursi(datei, true);
  }
  
  private void dateiSpeichernTursi(String datei, boolean nachHinten) {
      StringBuilder sb = new StringBuilder();
      sb.append("#!fill *\n");
      if(nachHinten) {
        sb.append("#!start startX\n");
      } else {
        sb.append("#!start " + this.start + "\n");
      }
      if (!Zustand.istZustand("S")) {
        System.err.println("Warnung: Keinen Endzustand S gefunden, "
            + "#!end-Zustand muss von Hand eingetragen werden");
        sb.append("#!end \n");
      } else {
        sb.append("#!end S\n");
      }
      this.dateiSpeichern(datei, 0, 1, 4, 2, 3
          , " ", " ", " ", " ", ""
          , "*", "-1", "1", "0" , nachHinten, sb.toString());      
  }
  
  /** Der uebergebene String wird als Beschreibung einer Turing-Maschine
   * interpretiert, die zeilenweise zur existierenden Turing-Maschine 
   * hinzugefuegt wird. Der Aufruf erfolgt typischerweise direkt nach der
   * Erstellung eines TuringMaschine-Objekts. 
   * @param gram Turing-Maschine in Textform
   */
  public void stringAlsTuringMaschine(String gram) {
    String[] zeilen = gram.split("\\n");
    this.ueber = new Ueberfuehrungsfunktion();
    this.zustaende = new ArrayList<>();
    Zeichen.reset();
    this.alphabet = new ArrayList<>();
    this.alphabet.add(Zeichen.zeichen(leerzeichen));
    for(String s:zeilen) {
      this.verarbeiteZeile(s);    
    }
  }
  
  private void verarbeiteZeile(String s) {
    String trim =s.split("//")[0].trim();
    if(trim.length() == 0) {
      return;
    }
    if(trim.startsWith("%")) {
      return;
    }
    if(trim.startsWith("Z:")) {
      this.neueZustaende(trim.substring(2).trim());
      return;
    }
    if(trim.startsWith("A:")) {
      this.neueZeichen(trim.substring(2).trim());
      return;
    }
    
    if(trim.startsWith("S:")) {
      if(!Zustand.istZustand(trim.substring(2).trim())) {
        throw new IllegalStateException("Startzszand " 
            + trim.substring(2).trim() + " nicht vorhanden");
      }
      this.start = Zustand.zustand(trim.substring(2).trim());
      return;
    }
    
    //Ueberfuehrungsfunktion
    String[] alle = trim.split("\\s+");
    if(alle.length != 5) {
      throw new IllegalStateException("Zeile " + trim + " fehlerhaft"
          + ", Ueberfuehrungsfunktion benoetigt 5 Elemente");
    } else {
      String richtung = alle[4];
      Richtung r = null;
      if(richtung.toLowerCase().equals("links") 
          || richtung.toLowerCase().equals("l")) {
        r = Richtung.LINKS;
      }
      if(richtung.toLowerCase().equals("rechts") 
          || richtung.toLowerCase().equals("r")) {
        r = Richtung.RECHTS;
      }
      if(richtung.toLowerCase().equals("stopp") 
          || richtung.toLowerCase().equals("s")) {
        r = Richtung.STOPP;
      }
      if(r == null) {
        throw new IllegalStateException(richtung + " kann nicht in eine Richtung "
            + "umgewandelt werden");
      } else {
        this.addUeberfuehrung(alle[0], alle[1], alle[2], alle[3], r);
      }
    }
    
  }
  
  private void neueZustaende(String s) {
    String[] alle = s.split("\\s+");
    for(String st:alle) {
      Zustand tmp = Zustand.zustand(st);
      if(!this.zustaende.contains(tmp)) {
        this.zustaende.add(tmp);
      }
    }
  }
  
  private void neueZeichen(String s) {
    String[] alle = s.split("\\s+");
    for(String st:alle) {
      Zeichen tmp = Zeichen.zeichen(st);
      if(!this.alphabet.contains(tmp)) {
        this.alphabet.add(tmp);
      }
    }
  }
  
  /** Setzt die spaeter abzuarbeitende Eingabe, die Abarbeitung kann
   * danach mit ausfuehren() gestartet werden.
   * @param init abzuarbeitende Eingabe
   */
  public void bearbeite(String init) {
    this.bearbeite(new Wort(init));
  }
  
  /** Setzt die spaeter abzuarbeitende Eingabe, die Abarbeitung kann
   * danach mit ausfuehren() gestartet werden.
   * @param init abzuarbeitendes eingegebenes Wort
   */
  public void bearbeite(Wort init) {
    if (this.modusRichtung == TMModusStart.HINTERDEMENDE) {
      this.konfiguration = new Konfiguration(this.start
          , init.vorne(LEERZEICHEN), LEERZEICHEN, new Wort());
    } else {
      if(init.laenge() == 0) {
        init = new Wort(LEERZEICHEN);
      }
      this.konfiguration = new Konfiguration(this.start
          , new Wort(LEERZEICHEN), init.at(0) 
          , init.anfuegen(LEERZEICHEN).loeschen(0));
    }
    this.haengt = false;
    this.halt = false;
    this.divergiert = false;
    this.bisher = new ArrayList<>();
  }
  
  /** Startet die Ausfuehrung der Turing-Maschine nachdem eine Eingabe
   * mit bearbeite() gesetzt wurde und berechnet das Ergebnis der
   * Turing-Maschine. Existiert keine Eingabe wird eine
   * IllegalStateException geworfen. Der Parameter gibt an, ob die 
   * Turing-Maschine die Ausfuehrung nach jedem Schritt unterbrechen und
   * auf Eingabe warten soll. 
   * @param interaktiv soll nach jedem Schritt auf eine Eingabe gewartet werden?
   * @return berechnetes Ergebnis
   */
  public Wort ausfuehren(boolean interaktiv) {
    if(this.konfiguration == null) {
      throw new IllegalStateException("Vor Berechnung ein Wort zur "
          + "Bearbeitung uebergeben");
    }
    System.out.println("Start: " + this.konfiguration);
    while(!this.haengt && !this.divergiert && !this.halt) {
      this.schritt();
      System.out.println(this.konfiguration);
      if(interaktiv) {
        System.out.println("Weiter mit <Return>");
        Eingabe.leseString();  
      }
    }
    System.out.println("Ende: " + this.konfiguration);
    Wort erg;
    if(this.modusRichtung == TMModusStart.HINTERDEMENDE) {
      erg = this.konfiguration.getLinks();
    } else {
      erg = this.konfiguration.getRechts().vorne(this.konfiguration.getKopf());
      while(erg.laenge() > 1 && erg.at(erg.laenge() - 1).equals(LEERZEICHEN)) {
        erg.loeschen(erg.laenge() - 1);
      }
    }
    return erg;
  }
  
  /** Turing-Maschine wird nach einer Eingabe (bearbeite()) 
   * schrittweise ausgefuehrt, dabei wird nach jedem Schritt auf 
   * eine Eingabe gewartet; am Ende wird das berechnte Wort zurueckgegeben,
   * insofern die Turing-Maschine terminiert.
   * @return berechnetes Wort
   */
  public Wort ausfuehrenSchrittweise() {
    return this.ausfuehren(true);
  }
  
  /** Turing-Maschine wird nach einer Eingabe (bearbeite()) 
   * ausgefuehrt; am Ende wird das berechnte Wort zurueckgegeben,
   * insofern die Turing-Maschine terminiert.
   * @return berechnetes Wort
   */
  public Wort ausfuehren() {
    return this.ausfuehren(false);
  }
  
  /** Berechnet ob die uebergebene Eingabe von der Turing-Maschine
   * akzeptiert wird, es besteht die Moeglichkeit zur Divergenz.
   * @param eingabe zu ueberpruefende Eingabe, ob diese zur akzeptierten
   *     Sprache der Turing-Maschine gehört 
   * @return gehört eingabe zur akzeptierten Sprache?
   */
  public boolean akzeptieren(String eingabe) {
    return this.akzeptieren(new Wort(eingabe));
  }
  
  /** Berechnet ob die uebergebene Eingabe von der Turing-Maschine
   * akzeptiert wird, es besteht die Moeglichkeit zur Divergenz.
   * @param eingabe zu ueberpruefende Eingabe, ob diese zur akzeptierten
   *     Sprache der Turing-Maschine gehört 
   * @return gehört eingabe zur akzeptierten Sprache?
   */
  public boolean akzeptieren(Wort eingabe) {
    try {
      this.berechnen(eingabe);
    } catch (Exception e) {
      // egal, da Turing-Maschine nur terminieren muss
    }
    return this.halt;
  }
  
  /** Berechnet das Ergebnis der Turing-Maschine, wenn diese auf 
   * die Eingabe angesetzt wird. Es besteht die Moeglichkeit 
   * zur Divergenz.
   * @param eingabe zu verarbeitende Eingabe
   * @return berechnetes Ergebnis
   */
  public Wort berechnen(String eingabe) {
    return this.berechnen(new Wort(eingabe));
  }
  
  /** Berechnet das Ergebnis der Turing-Maschine, wenn diese auf 
   * die Eingabe angesetzt wird. Es besteht die Moeglichkeit 
   * zur Divergenz.
   * @param eingabe zu verarbeitende Eingabe
   * @return berechnetes Ergebnis
   */
  public Wort berechnen(Wort eingabe) {
    return this.berechnen(1,1,new Wort[] {eingabe})[0];
  }
  
  /** Berechnet das Ergebnis der Turing-Maschine, wenn diese auf 
   * die Eingabe angesetzt wird, die aus definitionsbereich vielen
   * Wortern besteht und wertebereich viele Worte aks Ergebnis
   * berechnet. Dies entspricht einer Funktion mit
   * definitionsbereich-stelliger Eingabe und
   * werbereich-stelligem Ergebnis. Sollte die Eingabe nicht die
   * angegebene Elementanzahl haben, wird eine IllegalStateException
   * geworfen. Es besteht die Moeglichkeit zur Divergenz.
   * 
   * @param definitionsbereich Anzahl der zu uebergebene Worte als Eingabe
   * @param wertebereich Anzahl der berechneten Worte als Ergebnis
   * @param eingabe konkrete Eingabe, alls kommaseparierte Liste von Strings
   * @return berechnetes Ergebnis in einem Array
   * @throws IllegalStateException
   */
  public Wort[] berechnen(int definitionsbereich, int wertebereich
      , String... eingabe){
    Wort[] par = new Wort[eingabe.length];
    for(int i = 0; i < eingabe.length; i++) {
      par[i] = new Wort(eingabe[i]);
    }
    return this.berechnen(definitionsbereich, wertebereich, par);
  }
  
  /** Berechnet das Ergebnis der Turing-Maschine, wenn diese auf 
   * die Eingabe angesetzt wird, die aus definitionsbereich vielen
   * Wortern besteht und wertebereich viele Worte aks Ergebnis
   * berechnet. Dies entspricht einer Funktion mit
   * definitionsbereich-stelliger Eingabe und
   * werbereich-stelligem Ergebnis. Sollte die Eingabe nicht die
   * angegebene Elementanzahl haben, wird eine IllegalStateException
   * geworfen. Es besteht die Moeglichkeit zur Divergenz.
   * 
   * @param definitionsbereich Anzahl der zu uebergebene Worte als Eingabe
   * @param wertebereich Anzahl der berechneten Worte als Ergebnos
   * @param eingabe konkrete Eingabe
   * @return berechnetes Ergebnis in einem Array
   * @throws IllegalStateException
   */
  public Wort[] berechnen(int definitionsbereich, int wertebereich
      , Wort... eingabe){
    if(eingabe.length != definitionsbereich) {
      throw new IllegalStateException("erwartete Parameterzahl: " 
          + definitionsbereich + " gefunden: " + eingabe.length + ":"
          + Arrays.asList(eingabe));
    }
    StringBuilder sb = new StringBuilder();
    for(int i = 0; i < eingabe.length -1; i++) {
      sb.append(eingabe[i] + "#");
    }
    sb.append(eingabe[eingabe.length -1]);
    this.bearbeite(sb.toString());
   
    Wort erg = this.ausfuehren();
    if (this.divergiert || this.haengt || !this.halt) {
      return null;
    }

    if (erg.toString().startsWith("#")) {
      erg = erg.loeschen(0);
    }
    String[] tmp = erg.toString().split("#", -1);  // -1 fuer leeren String zwischen ##
    if(tmp.length - 0 != wertebereich) {
      throw new IllegalStateException("erwartete Ergebnisparameterzahl: " 
          + wertebereich + " gefunden: " + (tmp.length - 0) + " in "
          + erg);
    }
    Wort[] ergebnis = new Wort[wertebereich];
    for(int i = 0; i < tmp.length - 0 ; i++) {
      ergebnis[i] = new Wort(tmp[i]);
    }
    return ergebnis;
  }
  
  private TuringMaschine schritt() {
    if(this.halt) {
      System.err.println("Haltende Turing-Maschnie macht keinen Schritt mehr.");
      return this;
    }
    if(this.haengt) {
      System.err.println("Haengende Turing-Maschnine macht keinen Schritt mehr.");
      return this;
    }
    
    this.bisher.add(this.konfiguration.clone());
    Zustand neuZu = this.ueber.folgeZustand(this.konfiguration.getZustand()
        , this.konfiguration.getKopf());
    Zeichen neuZe = this.ueber.neuesZeichen(this.konfiguration.getZustand()
        , this.konfiguration.getKopf());
    Richtung ri = this.ueber.richtung(this.konfiguration.getZustand()
        , this.konfiguration.getKopf()); 
    this.konfiguration.setZustand(neuZu);
    
    switch(ri) {
      case STOPP: {
        this.halt = true;
        this.konfiguration.setKopf(neuZe);
        break;
      }
      case LINKS: {
        if(this.konfiguration.getLinks().equals(new Wort()) 
            && this.modusBand == TMModusBand.LINKSENDEND) {
          this.haengt = true;
          System.err.println("Warnung: Turingmaschine haengt bei Schritt "
              + "nach Links in " + this.konfiguration);
        } else {
          Wort lks = this.konfiguration.getLinks();
          if (lks.laenge() == 0) {
            lks = new Wort(LEERZEICHEN);
          }
          int lang = lks.laenge();
          this.konfiguration.setRechts( this.konfiguration
              .getRechts().vorne(neuZe));
          this.konfiguration.setKopf(lks.at(lang - 1));
          this.konfiguration.setLinks(lks.loeschen(lang - 1));
          break;
        }
      }
      case RECHTS: {
        Wort rts = this.konfiguration.getRechts();
        int lang = rts.laenge();
        this.konfiguration.setLinks( this.konfiguration
            .getLinks().anfuegen(neuZe));
        if (lang == 0) {
          this.konfiguration.setKopf(LEERZEICHEN);
        } else {
          this.konfiguration.setKopf(rts.at(0));
          this.konfiguration.setRechts(rts.loeschen(0));
        }
      }
    }
    if (this.bisher.contains(this.konfiguration) && ri != Richtung.STOPP) {
      this.divergiert = true;
      System.err.println("Warnung: Turing-Maschine terminiert nicht, wiederholt " 
          + this.konfiguration);
      
    }
    return this;
  }
  
  /** Fuegt zur Ueberfuehrungsfunktion eine Zeile hinzu, mit der im
   * Zustand alt mit gelesenenem Zeichen lesen in den Zustand neu
   * gegangen und schreiben auf das Band geschrieben wird, sowie der
   * Schreib-Lese-Kopf um r bewegt wird.
   * @param alt  Ausgangszustand
   * @param lesen gelesenes Zeichen
   * @param neu Folgezustand
   * @param schreiben geschriebenes Zeichen
   * @param r Bewegung des Schreib-Lese-Kopfes
   * @return diese Turing-Maschine ergaenzt um eine Zeile der
   *     Ueberfuehrungsfunktion
   */
  public TuringMaschine addUeberfuehrung(Zustand alt, Zeichen lesen
      , Zustand neu, Zeichen schreiben, Richtung r) { 
    this.ueber.add(alt, lesen, neu, schreiben, r);
    return this;
  }
  
  /** Fuegt zur Ueberfuehrungsfunktion eine Zeile hinzu, mit der im
   * Zustand alt mit gelesenenem Zeichen lesen in den Zustand neu
   * gegangen und schreiben auf das Band geschrieben wird, sowie der
   * Schreib-Lese-Kopf um r bewegt wird.
   * @param alt  Ausgangszustand
   * @param lesen gelesenes Zeichen
   * @param neu Folgezustand
   * @param schreiben geschriebenes Zeichen
   * @param r Bewegung des Schreib-Lese-Kopfes
   * @return diese Turing-Maschine ergaenzt um eine Zeile der
   *     Ueberfuehrungsfunktion
   */
  public TuringMaschine addUeberfuehrung(String alt, String lesen
      , String neu, String schreiben, Richtung r) { 
    if(!Zustand.istZustand(alt)) {
      throw new IllegalStateException("Zustand " + alt + " unbekannt");
    }
    if(!Zustand.istZustand(neu)) {
      throw new IllegalStateException("Zustand " + neu + " unbekannt");
    }
    if(!Zeichen.istZeichen(lesen)) {
      throw new IllegalStateException("Zeichen " + lesen + " unbekannt");
    }
    if(!Zeichen.istZeichen(schreiben)) {
      throw new IllegalStateException("Zeichen " + schreiben + " unbekannt");
    }
    this.ueber.add(Zustand.zustand(alt), Zeichen.zeichen(lesen)
        , Zustand.zustand(neu), Zeichen.zeichen(schreiben), r);
    return this;
  }
  
  /** Die Menge der Zustaende der Turing-Maschine wird um die mit
   * den uebergebenen namen ergaenzt
   * @param namen Namen der zu ergaenzenden Zustaende
   * @return diese Turing-Maschine mit neuen Zustaenden
   */
  public TuringMaschine addZustand(String... namen) {
    for(String s:namen) {
      Zustand tmp = Zustand.zustand(s);
      if (!this.zustaende.contains(tmp)) {
        this.zustaende.add(tmp);
      }
    }
    return this;
  }
  
  /** Das Alphabet der Turing-Maschine wird um die Zeichen mit
   * den uebergebenen namen ergaenzt
   * @param namen Namen der zu ergaenzenden Zeichen
   * @return diese Turing-Maschine mit neuen Zeichen
   */
  public TuringMaschine addZeichen(String... namen) {
    for(String s:namen) {
      Zeichen tmp = Zeichen.zeichen(s);
      if (!this.alphabet.contains(tmp)) {
        this.alphabet.add(tmp);
      }
    }
    for(String s:namen) {
      this.alphabet.add(Zeichen.zeichen(s));
    }
    return this;
  }

  public List<Zustand> getZustaende() {
    return zustaende;
  }

  public void setZustaende(List<Zustand> zustaende) {
    this.zustaende = zustaende;
  }

  public List<Zeichen> getAlphabet() {
    return alphabet;
  }

  public void setAlphabet(List<Zeichen> alphabet) {
    this.alphabet = alphabet;
  }

  public Zustand getStart() {
    return start;
  }

  public void setStart(Zustand start) {
    this.start = start;
  }

  public Ueberfuehrungsfunktion getUeber() {
    return ueber;
  }

  public void setUeber(Ueberfuehrungsfunktion ueber) {
    this.ueber = ueber;
  }

  public Konfiguration getKonfiguration() {
    return konfiguration;
  }

  public void setKonfiguration(Konfiguration konfiguration) {
    this.konfiguration = konfiguration;
  }

  public boolean isHalt() {
    return halt;
  }

  public void setHalt(boolean halt) {
    this.halt = halt;
  }

  public boolean isHaengt() {
    return haengt;
  }

  public void setHaengt(boolean haengt) {
    this.haengt = haengt;
  }

  public boolean isDivergiert() {
    return divergiert;
  }

  public void setDivergiert(boolean divergiert) {
    this.divergiert = divergiert;
  }

  public TMModusStart getModusRichtung() {
    return modusRichtung;
  }

  public void setModusRichtung(TMModusStart modusRichtung) {
    this.modusRichtung = modusRichtung;
  }

  public TMModusBand getModusBand() {
    return modusBand;
  }

  public void setModusBand(TMModusBand modusBand) {
    this.modusBand = modusBand;
  }

  public List<Konfiguration> getBisher() {
    return bisher;
  }

  public void setBisher(List<Konfiguration> bisher) {
    this.bisher = bisher;
  }

  @Override
  public int hashCode() {
    return Objects.hash(alphabet, start, ueber, zustaende);
  }

  @Override
  public boolean equals(Object obj) {
    if (this == obj)
      return true;
    if (obj == null)
      return false;
    if (getClass() != obj.getClass())
      return false;
    TuringMaschine other = (TuringMaschine) obj;
    return Objects.equals(alphabet, other.alphabet)
        && Objects.equals(start, other.start)
        && Objects.equals(ueber, other.ueber)
        && Objects.equals(zustaende, other.zustaende);
  }
  
  @Override
  public String toString() {
    return "Zustaende: " + this.zustaende 
        + "\nAlphabet: " + this.alphabet
        + "\nStart: " + this.start
        + "\n" + this.ueber;
  }
}
