package turingMaschine;

import java.io.BufferedWriter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import alphabet.Zeichen;
import util.DeepClone;
import util.Paar;
import zustand.Zustand;

public class Ueberfuehrungsfunktion implements DeepClone<Ueberfuehrungsfunktion>{

  private static final long serialVersionUID = 1L;
  // unschoene Aufteilung der Berechnung auf zwei Teilfunktionen, da
  // Klasse Tripel fehlt (evtl. aus 2 Paaren bauen)
  Map<Paar<Zustand, Zeichen>, Paar<Zustand, Zeichen>> funktion1 = new HashMap<>();
  Map<Paar<Zustand, Zeichen>, Richtung> funktion2 = new HashMap<>();
  
  /** Fuegt zur Ueberfuehrungsfunktion eine Zeile hinzu, mit der im
   * Zustand alt mit gelesenenem Zeichen lesen in den Zustand neu
   * gegangen und schreiben auf das Band geschrieben wird, sowie der
   * Schreib-Lese-Kopf um r bewegt wird.
   * @param alt  Ausgangszustand
   * @param lesen gelesenes Zeichen
   * @param neu Folgezustand
   * @param schreiben geschriebenes Zeichen
   * @param r Bewegung des Schreib-Lese-Kopfes
   * @return diese Turing-Maschine ergaenzt um eine Zeile der
   *     Ueberfuehrungsfunktion
   */
  public void add(Zustand alt, Zeichen lesen, Zustand neu, Zeichen schreiben, Richtung r) {
    this.funktion1.put(new Paar<>(alt,lesen), new Paar<>(neu, schreiben));
    this.funktion2.put(new Paar<>(alt,lesen), r);
  }
  
  /** Fuegt zur Ueberfuehrungsfunktion eine Zeile hinzu, mit der im
   * Zustand alt mit gelesenenem Zeichen lesen in den Zustand neu
   * gegangen und schreiben auf das Band geschrieben wird, sowie der
   * Schreib-Lese-Kopf um r bewegt wird.
   * @param alt  Ausgangszustand
   * @param lesen gelesenes Zeichen
   * @param neu Folgezustand
   * @param schreiben geschriebenes Zeichen
   * @param r Bewegung des Schreib-Lese-Kopfes
   * @return diese Turing-Maschine ergaenzt um eine Zeile der
   *     Ueberfuehrungsfunktion
   */
  public void add(String alt, String lesen, String neu, String schreiben
      , Richtung r) {
    this.add(Zustand.zustand(alt), Zeichen.zeichen(lesen)
        , Zustand.zustand(neu), Zeichen.zeichen(schreiben), r);
  }
  
  /** Berechnet den Nachfolgezustand fuer den gegebenen Zustand und
   * das eingelesene Zeichen.
   * 
   * @param alt Ausgangszustand
   * @param lesen auf Band gelesenes Zeichen
   * @return Folgezustand
   */
  public Zustand folgeZustand(Zustand alt, Zeichen lesen) {
    Paar<Zustand, Zeichen> ergebnis = this.funktion1.get(new Paar<>(alt, lesen));
    if (ergebnis == null) {
      throw new IllegalStateException("gibt keinen Nachfolgezustand zu " 
            + alt + "," + lesen);
    }
    return ergebnis.getLinks();
  }
  
  /** Berechnet das zu schreibende Zeichen fuer den gegebenen Zustand und
   * das eingelesene Zeichen.
   * 
   * @param alt Ausgangszustand
   * @param lesen auf Band gelesenes Zeichen
   * @return aud Band zu schreibendes Zeichen
   */
  public Zeichen neuesZeichen(Zustand alt, Zeichen lesen) {
    Paar<Zustand, Zeichen> ergebnis = this.funktion1.get(new Paar<>(alt, lesen));
    if (ergebnis == null) {
      throw new IllegalStateException("gibt keinen Nachfolgezustand zu " 
            + alt + "," + lesen);
    }
    return ergebnis.getRechts();
  }
  
  /** Berechnet die Richtung des Schreib-Lese-.Kopfes fuer den 
   * gegebenen Zustand und das eingelesene Zeichen.
   * 
   * @param alt Ausgangszustand
   * @param lesen auf Band gelesenes Zeichen
   * @return Richtung des Schreib-Lese-Kopfes
   */
  public Richtung richtung(Zustand alt, Zeichen lesen) {
    Richtung ergebnis = this.funktion2.get(new Paar<>(alt, lesen));
    if (ergebnis == null) {
      throw new IllegalStateException("gibt keinen Nachfolgezustand zu " 
            + alt + "," + lesen);
    }
    return ergebnis;
  }
  
  @Override
  public Ueberfuehrungsfunktion clone() {
    return this.deepClone();
  }

  @Override
  public int hashCode() {
    return Objects.hash(funktion1, funktion2);
  }

  @Override
  public boolean equals(Object obj) {
    if (this == obj)
      return true;
    if (obj == null)
      return false;
    if (getClass() != obj.getClass())
      return false;
    Ueberfuehrungsfunktion other = (Ueberfuehrungsfunktion) obj;
    return Objects.equals(funktion1, other.funktion1)
        && Objects.equals(funktion2, other.funktion2);
  }
  
  @Override
  public String toString() {
    StringBuilder sb = new StringBuilder("Ueberfuehrungsfunktion:\n");
    for(Paar<Zustand, Zeichen> alt: this.funktion1.keySet()) {
      Paar<Zustand, Zeichen> neu = this.funktion1.get(alt);
      sb.append(" ueber(" + alt.links +", " + alt.rechts + ") = ("
          + neu.getLinks() + ", " + neu.getRechts() + ", " 
          + this.funktion2.get(alt) +")\n");
    }
    return sb.toString();
  }

  /** Schreibt die Ueberfuehrungsfunktion in die als BufferdWriter
   * uebergebene geoeffnete Datei.
   * @param writer zu beschreibende geoeffnete Datei
   * @throws IOException
   */
  public void inDatei(BufferedWriter writer) throws IOException {
    @SuppressWarnings("unchecked")
    List<String>[] spalten = new List[5];
    int[] max = new int[5];
    for(int i = 0; i < 5; i++) {
      spalten[i] = new ArrayList<>();
      max[i] = 0;
    }
    
    for(Paar<Zustand, Zeichen> alt: this.funktion1.keySet()) {
     spalten[0].add(alt.getLinks().toString());
     spalten[1].add(alt.getRechts().toString());
     Paar<Zustand,Zeichen> neu = this.funktion1.get(alt);
     spalten[2].add(neu.getLinks().toString());
     spalten[3].add(neu.getRechts().toString());
     spalten[4].add(this.funktion2.get(alt).toString());
    }
    
    for(int i = 0; i < 5; i++) {
      for(String s: spalten[i]) {
        if(s.length() > max[i]) {
          max[i] = s.length();
        }
      }
    }
    
    for(int z = 0; z < spalten[0].size(); z++) {
      StringBuilder sb = new StringBuilder();
      for(int i = 0; i < 5; i++) {
        sb.append(spalten[i].get(z));
        sb.append(leerzeichen(max[i] - spalten[i].get(z).length() + 1));
      }
      sb.append("\n");
      writer.write(sb.toString());
    }   
  }
  
  /** Erzeugt einen String aus anzahl Leerzeichen.
   * 
   * @param anzahl der zurueckzugebenenden Leerzeichen
   * @return berechneter String aus Leerzeichen
   */
  public String leerzeichen(int anzahl) {
    StringBuilder sb = new StringBuilder();
    for( int i = 0; i < anzahl; i++) {
      sb.append(" ");
    }
    return sb.toString();
  }

  /** Schreibt die Ueberfuehrungsfunktion in die als BufferdWriter
   * uebergebene geoeffnete Datei im auf
   * turingmachinesimulator.com definierten Format.
   * @param writer zu beschreibende geoeffnete Datei
   * @throws IOException
   */
  public void inDateiUgarte(BufferedWriter writer) throws IOException {
    // vereinfachen, da aus Copy&Paste, spalten, max nicht benoetigt
    @SuppressWarnings("unchecked")
    List<String>[] spalten = new List[5];
    int[] max = new int[5];
    for(int i = 0; i < 5; i++) {
      spalten[i] = new ArrayList<>();
      max[i] = 0;
    }
    
    for(Paar<Zustand, Zeichen> alt: this.funktion1.keySet()) {
     spalten[0].add(alt.getLinks().toString());
     if (alt.getRechts().toString().equals("#")) {
       spalten[1].add("_");
     } else {
       spalten[1].add(alt.getRechts().toString());
     }
     Paar<Zustand,Zeichen> neu = this.funktion1.get(alt);
     spalten[2].add(neu.getLinks().toString());
     if (neu.getRechts().toString().equals("#")) {
       spalten[3].add("_");
     } else {
       spalten[3].add(neu.getRechts().toString());
     }
     spalten[4].add(this.funktion2.get(alt).toString());
    }
    
    for(int z = 0; z < spalten[0].size(); z++) {
      StringBuilder sb = new StringBuilder(); 
      sb.append(spalten[0].get(z) + "," + spalten[1].get(z) + "\n");
      sb.append(spalten[2].get(z) + "," + spalten[3].get(z) + ",");
      switch(spalten[4].get(z)) {
        case "LINKS": {
          sb.append("<\n");
          break;
        }
        case "RECHTS": {
          sb.append(">\n");
          break;
        }
        case "STOPP":{
          sb.append("-\n");
          break;
        }
        default: {
          System.err.println("Warnung: Ueberfuehrungsfunktion hat nicht "
              + "erwartetes Format: " + spalten[4].get(z));
        }
      }
      sb.append("\n");
      writer.write(sb.toString());
    }   
  }

  /** Methode wird intern genutzt, um unterschiedliche Datenformate
   * auf Basis der vorhandenen Daten zu generieren. Es werden neben 
   * der geoeffneten Datei u. a. die Nummern der Spalten uebergeben, die die
   * Reihenfolge in der Ausgabe festlegt. Dazu muessen die Zahlen
   * 0 bis 4 vergeben werden. 
   * @param datei zum Schreiben geoeffnete Datei
   * @param alt Nummer der Spalte des ausgehenden Zustands
   * @param lesen Nummer der Spalte des eingelesenen Zeichens
   * @param neu Nummer der Spalte des neu erreichten Zustands
   * @param schreiben Nummer der Spalte des geschriebenen Zeichens
   * @param richtung Nummer der Spalte der Bewegung der TM
   * @param trenn1 Trennsymbol nach der ersten Ausgabe
   * @param trenn2 Trennsymbol nach der zweiten Ausgabe
   * @param trenn3 Trennsymbol nach der dritten Ausgabe
   * @param trenn4 Trennsymbol nach der vierten Ausgabe
   * @param trenn5 Trennsymbol nach der fuenften Ausgabe, z. B. Zeilentrenner
   * @param leerzeichen zu verwendendes Leerzeichen
   * @param lks String mit dem TM nach links laufen soll
   * @param rts String mit dem TM nach rechts lauifen soll
   * @param stopp String mit dem die TM stehenbleibt
   * @param nachHinten String der am Ende der Datei angefuegt wird
   * @param amAnfang String der am Anfang der Ausgabe geschrieben wird 
   */
  public void inDateiFormatiert(BufferedWriter writer, int altsp
      , int lesen, int neusp, int schreiben, int richtung, String trenn1
      , String trenn2, String trenn3, String trenn4, String trenn5
      , String leerzeichen, String lks, String rts, String stopp)
          throws IOException {
    @SuppressWarnings("unchecked")
    List<String>[] spalten = new List[5];
    int[] max = new int[5];
    for(int i = 0; i < 5; i++) {
      spalten[i] = new ArrayList<>();
      max[i] = 0;
    }
    
    for(Paar<Zustand, Zeichen> alt: this.funktion1.keySet()) {
     spalten[altsp].add(alt.getLinks().toString());
     if (alt.getRechts().equals(TuringMaschine.LEERZEICHEN)) {
       spalten[lesen].add(leerzeichen);
     } else {
       spalten[lesen].add(alt.getRechts().toString());
     }
     Paar<Zustand,Zeichen> neu = this.funktion1.get(alt);
     spalten[neusp].add(neu.getLinks().toString());
     if (neu.getRechts().equals(TuringMaschine.LEERZEICHEN)) {
       spalten[schreiben].add(leerzeichen);
     } else {
       spalten[schreiben].add(neu.getRechts().toString());
     }
     spalten[richtung].add(this.funktion2.get(alt).toString());
    }
    String[] trenner = {trenn1, trenn2, trenn3, trenn4, trenn5};
    
    for(int z = 0; z < spalten[0].size(); z++) {
      StringBuilder sb = new StringBuilder();
      for(int zz = 0;  zz < 5; zz++) {
        if (zz == richtung) {
          switch(spalten[richtung].get(z)) {
            case "LINKS": {
              sb.append(lks + trenner[zz]);
              break;
            }
            case "RECHTS": {
              sb.append(rts + trenner[zz]);
              break;
            }
            case "STOPP":{
              sb.append(stopp + trenner[zz]);
              break;
            }
            default: {
              System.err.println("Warnung: Ueberfuehrungsfunktion hat nicht "
                  + "erwartetes Format: " + spalten[zz].get(z));
            }
          }
        } else {
          sb.append(spalten[zz].get(z) + trenner[zz]);
        }
      }
      sb.append("\n");
      writer.write(sb.toString());
    } 
  }
    
}
