package tabelle;

import java.beans.XMLDecoder;
import java.beans.XMLEncoder;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.util.ArrayList;
import java.util.List;

/**
 * Klasse Tabelle beschreibt eine sehr einfache Tabellenform,
 * die fuer jede Spalte eine Ueberschrift hat und alle Inhalte
 * vom Typ Object speichert. Zahlen werden dazu in inhren passenden
 * Typen, z. B. int nach Integer umgewandelt.
 * @author x
 */
public class Tabelle {
    private String name = "";
    private String[] spaltenueberschriften = {};
    private List<Object[]> inhalt;
    
    
    public Tabelle(){
        this.inhalt = new ArrayList<>();
    }
    
    public Tabelle(String name, String[] spaltenueberschrift){
        this();
        this.name = name;
        this.spaltenueberschriften = spaltenueberschrift;
        
    }
    
    public void zeileHinzu(Object[] werte){
        this.pruefeUeberschriften();
        if(this.spaltenueberschriften.length != werte.length){
            throw new IllegalArgumentException("Zeile muss aus "
                    + this.spaltenueberschriften.length + "Elementen bestehen "
                    + "und nicht " + werte.length);
        }
        this.inhalt.add(werte);               
    }
    
    private void pruefeUeberschriften(){
        if(this.spaltenueberschriften == null 
                || this.spaltenueberschriften.length ==0){
            throw new IllegalArgumentException(
                    "keine Spaltenueberschriften definiert");
        }
    }
    
    /**
     * Gibt die Anzahl der vorhandenen Zeilen zurueck.
     * @return Anzahl Zeilen
     */
    public int anzahlZeilen(){
        return this.inhalt.size();
    }
    
    /**
     * Gibt das Objekt der zeile Zeile aus der Spalte mit der Ueberschrift
     * spalte zurueck. Sollte die zeile oder die Spaltenueberschrift nicht
     * existieren wird eine IllegalArgumentException geworfen.
     * @param zeile Wert in dieser Zeile gesucht
     * @param spalte Aus aus der Spalte mit diesem Namen gesucht
     * @return gefundener Wert in der Zeile zeile und der Spalte mit Namen 
     * spalte
     */
    public Object wertInZeileVon(int zeile, String spalte){
        if (zeile >= this.anzahlZeilen()){
            throw new IllegalArgumentException("Zeile " + zeile 
                    + " existiert nicht");
        }
        int spaltennr = this.spaltennummer(spalte);
      return this.inhalt.get(zeile)[spaltennr];
    }
    
    private int spaltennummer(String spalte){
        for (int i=0; i < this.spaltenueberschriften.length; i++){
            if(spalte.equals(this.spaltenueberschriften[i])){
                return i;
            }
        }
        throw new IllegalArgumentException("Spalte mit namen \"" + spalte
                + " existiert nicht");
    }

    /**
     * Gibt als Array alle Spaltenueberschriften zurueck.
     * @return Array mit Spaltenueberschriften
     */
    public String[] getSpaltenueberschriften() {
        return spaltenueberschriften;
    }

    public String getName() {
        return name;
    }
    
    public void speichern (String dateiname){
        try (XMLEncoder out = new XMLEncoder(
                new BufferedOutputStream(new FileOutputStream(dateiname)))) {
            out.writeObject(this.name);
            out.writeObject(this.spaltenueberschriften);
            out.writeObject(this.inhalt);
        } catch (FileNotFoundException e) {
            System.err.println("Speichern gescheitert.");
        }
    }
    
    @SuppressWarnings("unchecked")
    public void laden(String dateiname){
        try (XMLDecoder in = new XMLDecoder(
                new BufferedInputStream(new FileInputStream(dateiname)))) {
            this.name = (String) in.readObject();
            this.spaltenueberschriften = (String[]) in.readObject();
            this.inhalt = (List<Object[]>) in.readObject();
        } catch (FileNotFoundException e) {
            throw new IllegalArgumentException("Datei " + dateiname 
                    + " nicht gefunden");
        } catch (Exception e) {
            throw new IllegalArgumentException("Probleme beim laden: " 
                    + e.getMessage());
        }
    }
    
    public void zeigen(){
        int[] spaltenbreite = new int[this.spaltenueberschriften.length];
        for(int i=0; i < this.spaltenueberschriften.length; i++){
            spaltenbreite[i] = this.spaltenueberschriften[i].length();
        }
        for(Object[] zeile: this.inhalt){
            for(int s = 0; s < zeile.length; s++){
                int laenge = zeile[s].toString().length();
                if (laenge > spaltenbreite[s]){
                    spaltenbreite[s] = laenge;
                }
            }
        }
        
        this.fuellzeichen(this.name.length() + 4,'#');
        System.out.print("\n# ");
        System.out.println(this.name + " #");
        this.fuellzeichen(this.name.length() + 4,'#');
        System.out.println("");
        
        for(int i=0; i < this.spaltenueberschriften.length; i++){
            System.out.print(this.spaltenueberschriften[i]);
            this.fuellzeichen(spaltenbreite[i] 
                    - this.spaltenueberschriften[i].length(),' ');
            System.out.print("|");
        }
        System.out.println("");
        
        for(int i=0; i < this.spaltenueberschriften.length; i++){
            this.fuellzeichen(spaltenbreite[i],'-');
            System.out.print("+");
        }
        System.out.println("");
        
        for(Object[] zeile: this.inhalt){
            for(int i=0; i < this.spaltenueberschriften.length; i++){
                System.out.print(zeile[i]);
                this.fuellzeichen(spaltenbreite[i] 
                        - zeile[i].toString().length(), '.');
                System.out.print("|");
            }
            System.out.println("");
        }
        
    }
    
    private void fuellzeichen(int anzahl, char zeichen){
        for(int i=0; i < anzahl; i++){
            System.out.print(zeichen);
        }
    }
}
